unit MPEAnimations;

interface

// -----------------------------------------------------------------------------
uses
  Windows, SysUtils, Classes, Graphics,
  Math,
  MPEBitmap, MPEExt;

// -----------------------------------------------------------------------------
type
  TBitmapChannel = (bcA, bcR, bcG, bcB, bcRGB, bcARGB);
  
  TMPEAnimations = class(TObject)
  private
    // Parametry
    FPosition: Integer;

    FFrames: array[0..2] of TList;

    // Metody parametrow
    function GetCount: Integer;
    procedure SetPosition(Position: Integer);
    function GetFrame(Axis: Byte; Index: Integer): TMPEBitmap;

  public
    // Parametry
    property Count: Integer read GetCount;
    property Position: Integer read FPosition write SetPosition;

    property Frames[Axis: Byte; Index: Integer]: TMPEBitmap read GetFrame;

    // Metody
    function AddFrame(Axis: Byte): TMPEBitmap;
    procedure DeleteFrame(Axis: Byte; Index: Integer);
    procedure DeleteAllFrames;

    procedure ImportFromDirectory(const Directory: String);
    procedure ImportFromProjectDirectory;

    procedure Draw(Axis: Byte; Canvas: TCanvas;
      X, Y, Width, Height: Integer);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;

  end;

implementation

// -----------------------------------------------------------------------------
uses
  MPEProject;

// -----------------------------------------------------------------------------
function TMPEAnimations.GetCount: Integer;
begin
Result := Min(Min(FFrames[0].Count, FFrames[1].Count), FFrames[2].Count);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.SetPosition(Position: Integer);
begin

if FFrames[0].Count > 0 then
  FPosition := BoundsRange(Position, 0, Count-1)
else
  FPosition := -1;

end;

// -----------------------------------------------------------------------------
function TMPEAnimations.GetFrame(Axis: Byte; Index: Integer): TMPEBitmap;
begin
Result := TMPEBitmap(FFrames[Axis][Index]);
end;

// -----------------------------------------------------------------------------
function TMPEAnimations.AddFrame(Axis: Byte): TMPEBitmap;
begin
Result := TMPEBitmap.Create;
FFrames[Axis].Add(Result);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.DeleteFrame(Axis: Byte; Index: Integer);
begin
TMPEBitmap(FFrames[Axis][Index]).Free;
FFrames[Axis].Delete(Index);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.DeleteAllFrames;
var
  b: Byte;
  i: Integer;
begin

// Zwalnianie klatek
for b := 0 to Length(FFrames)-1 do
  begin

  if FFrames[b].Count > 0 then
    for i := 0 to FFrames[b].Count-1 do
      TMPEBitmap(FFrames[b][i]).Free;

  FFrames[b].Clear;

  end;

// Pozycja
FPosition := -1;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ImportFromDirectory(const Directory: String);
var
  searchrec: TSearchRec;
  frame_filename_array: array[0..3] of String;
  frame: TMPEBitmap;
begin

// Czyszczenie klatek
DeleteAllFrames;

// Odczytywanie plikow
if DirectoryExists(Directory) then
  begin

  if FindFirst(Project.Path + Project.SimName + '.*.*' + GRAPHICS_OUTPUT_DATA_FILEEXT, faAnyFile, searchrec) = 0 then
    repeat
      Split('.', searchrec.Name, frame_filename_array);
      frame := AddFrame(StrToInt(frame_filename_array[2]));
      frame.LoadFromFileEx(Project.Path + searchrec.Name);
    until not (FindNext(searchrec) = 0);

  FindClose(searchrec);

  end;

// Pozycja animacji
if Count > 0 then
  FPosition := 0
else
  FPosition := -1;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ImportFromProjectDirectory;
begin
ImportFromDirectory(Project.Path);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.Draw(Axis: Byte; Canvas: TCanvas;
  X, Y, Width, Height: Integer);
var
  frame: TMPEBitmap;
  buffer: TBitmap;
begin

// Klatka
if (FPosition >= 0) and (FPosition <= FFrames[Axis].Count-1) then
  frame := Frames[Axis, FPosition]
else
  frame := nil;

// Bufor
buffer := TBitmap.Create;  // TODO: Utworzyc globalnie i tylko czyscic
buffer.Canvas.Brush.Color := clWhite;
buffer.PixelFormat := pf24bit;
buffer.Width := Width;
buffer.Height := Height;

// Zapelnianie bufora
if Assigned(frame) then
  buffer.Canvas.CopyRect(
    Bounds((Width-frame.Width) div 2, (Height-frame.Height) div 2, frame.Width, frame.Height),
    frame.Canvas,
    Bounds(0, 0, frame.Width, frame.Height)
    );

// Zrzut bufora
Canvas.CopyRect(
  Bounds(X, Y, Width, Height),
  buffer.Canvas,
  Bounds(0, 0, Width, Height)
  );

// Zwolnienie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEAnimations.Create;
var
  b: Byte;
begin

inherited Create;

// Parametry
FPosition := -1;

for b := 0 to Length(FFrames)-1 do
  FFrames[b] := TList.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEAnimations.Destroy;
var
  b: Byte;
begin

// Zwalnianie klatek
DeleteAllFrames;

// Parametry
for b := 0 to Length(FFrames)-1 do
  FFrames[b].Free;

inherited Destroy;

end;

end.
