unit MPEBuffer;

interface

// -----------------------------------------------------------------------------
uses
  Windows, SysUtils, Classes,
  ULZMADecoder, ULZMAEncoder, ZLib;

// -----------------------------------------------------------------------------
type
  TCompressMethod = (cmNone, cmZLib, cmLZMA);

  TMPEBuffer = class(TObject)
  const
    // Stale
    BOOLEAN_LEN    = 1;
    BYTE_LEN       = 1;
    SHORTINT_LEN   = 1;
    WORD_LEN       = 2;
    INTEGER_LEN    = 4;
    DOUBLE_LEN     = 8;
    PROPERTIES_LEN = 5;

  private
    // Parametry
    FStream: TStream;

    // Metody parametrow
    function GetPosition: Integer;
    procedure SetPosition(Position: Integer);
    function GetSize: Integer;
    procedure SetSize(Size: Integer);

  public
    // Parametry
    property Position: Integer read GetPosition write SetPosition;
    property Size: Integer read GetSize write SetSize;

    property Stream: TStream  read FStream;

    // Metody
    function ReadBoolean: Boolean;
    function ReadByte: Byte;
    function ReadShortint: Shortint;
    function ReadWord: Word;
    function ReadInteger: Integer;
    function ReadDouble: Double;
    function ReadString: String;
    function ReadStream(Stream: TStream): Integer;
    function ReadBuffer(var Buffer): Integer;

    procedure WriteData(Data: Boolean); overload;
    procedure WriteData(Data: Byte); overload;
    procedure WriteData(Data: Shortint); overload;
    procedure WriteData(Data: Word); overload;
    procedure WriteData(Data: Integer); overload;
    procedure WriteData(Data: Double); overload;
    procedure WriteData(const Data: String); overload;
    procedure WriteData(Data: TStream;
      CompressMethod: TCompressMethod = cmNone); overload;
    procedure WriteData(const Buffer; Count: Integer;
      CompressMethod: TCompressMethod = cmNone); overload;

    procedure Skip(Size: Integer);
    procedure SkipBoolean;
    procedure SkipByte;
    procedure SkipShortint;
    procedure SkipWord;
    procedure SkipInteger;
    procedure SkipString;
    procedure SkipStream;
    procedure SkipBuffer;

    // Tworzenie
    constructor Create(Stream: TStream);
    
  end;

implementation

// -----------------------------------------------------------------------------
function TMPEBuffer.GetPosition: Integer;
begin
Result := FStream.Position;
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SetPosition(Position: Integer);
begin
FStream.Position := Position;
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.GetSize: Integer;
begin
Result := FStream.Size;
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SetSize(Size: Integer);
begin
FStream.Size := Size;
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadBoolean: Boolean;
begin
FStream.Read(Result, BOOLEAN_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadByte: Byte;
begin
FStream.Read(Result, BYTE_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadShortint: Shortint;
begin
FStream.Read(Result, SHORTINT_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadWord: Word;
begin
FStream.Read(Result, WORD_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadInteger: Integer;
begin
FStream.Read(Result, INTEGER_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadDouble: Double;
begin
FStream.Read(Result, DOUBLE_LEN);
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadString: String;
var
  len: Word;
  buf: PByteArray;
begin

// Dlugosc lancucha
len := ReadWord;

// Bufor lancucha
buf := AllocMem(len);

// Dane bufora
FStream.Read(buf^, len);

// Dane lanucha
SetLength(Result, len);
CopyMemory(@Result[1], buf, len);

// Zwalnianie
FreeMem(buf);
          
end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadStream(Stream: TStream): Integer;
var
  compress: TCompressMethod;
  zlib: TDecompressionStream;
  lzma: TLZMADecoder;
  properties: array[0..4] of Byte;
begin

// Dane kompresji
compress := TCompressMethod(ReadByte);

// Dlugosc strumienia
Stream.Position := 0;
Result := ReadInteger;

// Dane strumienia
case compress of
  cmNone:
    if Result > 0 then
      Stream.CopyFrom(FStream, Result);
  cmZLib:
    begin
    zlib := TDecompressionStream.Create(FStream);
    Stream.CopyFrom(zlib, Result);
    zlib.Free;
    end;
  cmLZMA:
    begin
    FStream.Read(properties, PROPERTIES_LEN);
    lzma := TLZMADecoder.Create;
    lzma.SetDecoderProperties(properties);
    lzma.Code(FStream, Stream, Result);
    lzma.Free;
    end;
end;

// Inne
Stream.Position := 0;

end;

// -----------------------------------------------------------------------------
function TMPEBuffer.ReadBuffer(var Buffer): Integer;
var
  stream: TMemoryStream;
begin

// Strumien
stream := TMemoryStream.Create;

// Dane bufora
Result := ReadStream(stream);
stream.Read(Buffer, Result);

// Zwolnienie
stream.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Boolean);
begin
FStream.Write(Data, BOOLEAN_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Byte);
begin
FStream.Write(Data, BYTE_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Shortint);
begin
FStream.Write(Data, SHORTINT_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Word);
begin
FStream.Write(Data, WORD_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Integer);
begin
FStream.Write(Data, INTEGER_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: Double);
begin
FStream.Write(Data, DOUBLE_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(const Data: String);
var
  len: Word;
  buf: PByteArray;
begin

// Dlugosc lancucha
len := Length(Data);
WriteData(len);

// Bufor lancucha
buf := AllocMem(len);
CopyMemory(buf, @Data[1], len);

// Dane bufora
FStream.Write(buf^, len);

// Zwalnianie
FreeMem(buf);

end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(Data: TStream;
  CompressMethod: TCompressMethod = cmNone);
var
  len: Integer;
  zlibstream: TCompressionStream;
  lzma: TLZMAEncoder;
begin

// Dane kompresji
WriteData(Byte(CompressMethod));

// Dlugosc strumienia
Data.Position := 0;
len := Data.Size;
WriteData(len);

// Dane skompresowanego strumienia
case CompressMethod of
  cmNone:
    FStream.CopyFrom(Data, 0);
  cmZLib:
    begin
    zlibstream := TCompressionStream.Create(clMax, FStream);
    zlibstream.CopyFrom(Data, len);
    zlibstream.Free;
    end;
  cmLZMA:
    begin
    lzma := TLZMAEncoder.Create;
    lzma.WriteCoderProperties(FStream);
    lzma.Code(Data, FStream, -1, -1);
    lzma.Free;
    end;
end;

// Inne
Data.Position := 0;

end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.WriteData(const Buffer; Count: Integer;
  CompressMethod: TCompressMethod = cmNone);
var
  stream: TMemoryStream;
begin

// Strumien
stream := TMemoryStream.Create;

// Dane bufora
stream.Write(Buffer, Count);
WriteData(stream, CompressMethod);

// Zwolnienie
stream.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.Skip(Size: Integer);
begin
FStream.Seek(Size, soCurrent);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipBoolean;
begin
Skip(BOOLEAN_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipByte;
begin
Skip(BYTE_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipShortint;
begin
Skip(SHORTINT_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipWord;
begin
Skip(WORD_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipInteger;
begin
Skip(INTEGER_LEN);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipString;
begin
Skip(ReadWord);
end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipStream;
var
  compress: TCompressMethod;
begin

// Metoda kompresji
compress := TCompressMethod(ReadByte);

// Pominiecie
Skip(ReadInteger);

if compress = cmLZMA then
  Skip(PROPERTIES_LEN);

end;

// -----------------------------------------------------------------------------
procedure TMPEBuffer.SkipBuffer;
begin
SkipStream;
end;

// -----------------------------------------------------------------------------
constructor TMPEBuffer.Create(Stream: TStream);
begin

inherited Create;

// Parametry
FStream := Stream;

end;

end.
