unit MPEGraph;

interface

// -----------------------------------------------------------------------------
uses
  Windows, Classes, SysUtils, Graphics, ExtCtrls,
  Math, Resample,
  MPEBuffer, MPEExt, MPEOutputData;

// -----------------------------------------------------------------------------
type
  TMPEDataPoint = class(TObject)
  private
    // Parametry
    FX, FY: Double;

  public
    // Parametry
    property X: Double read FX write FX;
    property Y: Double read FY write FY;

    // Metody
    procedure Assign(DataPoint: TMPEDataPoint);

    // Tworzenie
    constructor Create;
    
  end;

  TMPEDataSeries = class(TObject)
  private
    // Parametry
    FName: String;
    FColor: Integer;
    FLineWidth: Byte;

    FAxisXColumn, FAxisYColumn: Byte;

    FDataPoints: TList;

    // Metody parametrow
    function GetDataPoint(Index: Integer): TMPEDataPoint;
    function GetDataPointsCount: Integer;

  public
    // Parametry
    property Name: String read FName write FName;
    property Color: Integer read FColor write FColor;
    property LineWidth: Byte read FLineWidth write FLineWidth;

    property AxisXColumn: Byte read FAxisXColumn write FAxisXColumn;
    property AxisYColumn: Byte read FAxisYColumn write FAxisYColumn;

    property DataPoints[Index: Integer]: TMPEDataPoint read GetDataPoint;
    property DataPointsCount: Integer read GetDataPointsCount;

    // Metody
    function AddDataPoint(const X, Y: Double): TMPEDataPoint;
    procedure DeleteDataPoint(Index: Integer);
    procedure DeleteAllDataPoints;
    procedure SortDataPoints;

    procedure ImportFromOutputData(OutputData: TMPEOutputData);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;
  
  end;

  TMPEGraph = class(TObject)
  private
    // Parametry
    FName: String;
    FAxesColor: Integer;
    FBackgroundColor: Integer;
    FFontColor: Integer;
    FRepaintTime: Word;
    FSamples: Byte;
    FWidth: Integer;
    FHeight: Integer;

    FAxisXAutomatic: Boolean;
    FAxisXMin, FAxisXMax: Double;
    FAxisYAutomatic: Boolean;
    FAxisYMin, FAxisYMax: Double;

    FDataSeries: TList;

    FBitmap: TBitmap;
    FImage: TImage;

    // Metody parametrow
    procedure SetWidth(Width: Integer);
    procedure SetHeight(Height: Integer);

    function GetDataSeries(Index: Integer): TMPEDataSeries;
    function GetDataSeriesCount: Integer;

    // Metody
    function ValueToPixels(X, Y, XMin, XMax, YMin, YMax: Double): TPoint;
    procedure PixelsToValue(Left, Top: Integer;
      XMin, XMax, YMin, YMax: Double; var X, Y: Double);
    function AxesToPixels: TPoint;
    function PointToPixels(DataSeriesIndex, DataPointIndex: Integer): TPoint;

    procedure FitAxes(Margin: Single = 0);

  public
    // Parametry
    property Name: String read FName write FName;
    property AxesColor: Integer read FAxesColor write FAxesColor;
    property BackgroundColor: Integer read FBackgroundColor write FBackgroundColor;
    property FontColor: Integer read FFontColor write FFontColor;
    property RepaintTime: Word read FRepaintTime;
    property Samples: Byte read FSamples write FSamples;
    property Width: Integer read FWidth write SetWidth;
    property Height: Integer read FHeight write SetHeight;

    property AxisXAutomatic: Boolean read FAxisXAutomatic write FAxisXAutomatic;
    property AxisXMin: Double read FAxisXMin write FAxisXMin;
    property AxisXMax: Double read FAxisXMax write FAxisXMax;
    property AxisYAutomatic: Boolean read FAxisYAutomatic write FAxisYAutomatic;
    property AxisYMin: Double read FAxisYMin write FAxisYMin;
    property AxisYMax: Double read FAxisYMax write FAxisYMax;

    property DataSeries[Index: Integer]: TMPEDataSeries read GetDataSeries;
    property DataSeriesCount: Integer read GetDataSeriesCount;

    property Bitmap: TBitmap read FBitmap;
    property Image: TImage read FImage write FImage;

    // Metody
    procedure PixelsToPoint(Left, Top: Integer; var X, Y: Double);

    function AddDataSeries: TMPEDataSeries;
    procedure AssignDataSeries(DataSeries: TMPEDataSeries);
    procedure DeleteDataSeries(Index: Integer);
    procedure DeleteAllDataSeries;

    procedure Repaint;

    procedure ExportToFile(const Filename: String);

    procedure LoadFromStream(Stream: TStream);
    procedure SaveToStream(Stream: TStream);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;

  end;

implementation

// -----------------------------------------------------------------------------
procedure TMPEDataPoint.Assign(DataPoint: TMPEDataPoint);
begin
FX := DataPoint.X;
FY := DataPoint.Y;
end;

// -----------------------------------------------------------------------------
constructor TMPEDataPoint.Create;
begin

inherited;

// Parametry
FX := 0;
FY := 0;

end;



// -----------------------------------------------------------------------------
function TMPEDataSeries.GetDataPoint(Index: Integer): TMPEDataPoint;
begin
Result := FDataPoints[Index];
end;

// -----------------------------------------------------------------------------
function TMPEDataSeries.GetDataPointsCount: Integer;
begin
Result := FDataPoints.Count;
end;

// -----------------------------------------------------------------------------
function TMPEDataSeries.AddDataPoint(const X, Y: Double): TMPEDataPoint;
begin
Result := TMPEDataPoint.Create;
Result.X := X;
Result.Y := Y;
FDataPoints.Add(Result);
end;

// -----------------------------------------------------------------------------
procedure TMPEDataSeries.DeleteDataPoint(Index: Integer);
begin
DataPoints[Index].Free;
FDataPoints.Delete(Index);
end;

// -----------------------------------------------------------------------------
procedure TMPEDataSeries.DeleteAllDataPoints;
var
  c: Integer;
begin

// Zwalnianie punktow
if FDataPoints.Count > 0 then
  for c := 0 to FDataPoints.Count-1 do
    TMPEDataPoint(FDataPoints[c]).Free;

// Czyszczenie listy
FDataPoints.Clear;

end;

// -----------------------------------------------------------------------------
procedure TMPEDataSeries.SortDataPoints;

  // ---------------------------------------------------------------------------
  procedure Sort(ilow, ihigh: Integer);
  var
    ilo, ihi: Integer;
    d: Double;
    t: TMPEDataPoint;
  begin

  t := TMPEDataPoint.Create;

  ilo := ilow;
  ihi := ihigh;

  d := DataPoints[(ilow + ihigh) div 2].X;

  repeat

    while DataPoints[ilo].X < d do Inc(ilo);
    while DataPoints[ihi].X > d do Dec(ihi);

    if (ilo <= ihi) then
      begin
      t.Assign(DataPoints[ilo]);
      DataPoints[ilo].Assign(DataPoints[ihi]);
      DataPoints[ihi].Assign(t);

      Inc(ilo);
      Dec(ihi);
      end;

  until ilo > ihi;

  if (ihi > ilow) then Sort(ilow, ihi);
  if (ilo < ihigh) then Sort(ilo, ihigh);

  t.Free;

  end;
  // ---------------------------------------------------------------------------

begin

if DataPointsCount > 0 then
  Sort(0, DataPointsCount-1);

end;

// -----------------------------------------------------------------------------
procedure TMPEDataSeries.ImportFromOutputData(OutputData: TMPEOutputData);
var
  i: Integer;
begin

if not (OUTPUT_DATA_NONE_COLUMN in [FAxisXColumn, FAxisYColumn]) then
  if OutputData.Count > 0 then
    begin

    for i := 0 to OutputData.Count-1 do
      AddDataPoint(
        OutputData.Data[i, FAxisXColumn],
        OutputData.Data[i, FAxisYColumn]
        );

    //SortDataPoints;

    end;

end;

// -----------------------------------------------------------------------------
constructor TMPEDataSeries.Create;
begin

inherited;

// Parametry
FName := '';
FColor := $000000;
FLineWidth := 1;

FAxisXColumn := OUTPUT_DATA_NONE_COLUMN;
FAxisYColumn := OUTPUT_DATA_NONE_COLUMN;

FDataPoints := TList.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEDataSeries.Destroy;
begin

// Zwalnianie
DeleteAllDataPoints;

// Parametry
FDataPoints.Free;   

inherited;

end;



// -----------------------------------------------------------------------------
procedure TMPEGraph.SetWidth(Width: Integer);
begin
FWidth := Width;
FBitmap.Width := FWidth;
end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.SetHeight(Height: Integer);
begin
FHeight := Height;
FBitmap.Height := FHeight;
end;

// -----------------------------------------------------------------------------
function TMPEGraph.GetDataSeries(Index: Integer): TMPEDataSeries;
begin
Result := FDataSeries[Index];
end;

// -----------------------------------------------------------------------------
function TMPEGraph.GetDataSeriesCount: Integer;
begin
Result := FDataSeries.Count;
end;

// -----------------------------------------------------------------------------
function TMPEGraph.ValueToPixels(X, Y, XMin, XMax, YMin, YMax: Double): TPoint;
var
  ax, ay{, dv}: Double;
begin

// Pozycje wzgledne
ax := (X-XMin) / (XMax-XMin);
ay := 1-((Y-YMin) / (YMax-YMin));
                    {
// Ograniczanie wartosci wzglednych
if (ax >= 0) and (ax <= 1) and (ay >= 0) and (ay <= 1) then
  dv := 1
else if (ax > 1) and (ay > 1) then
  dv := Min(ax, ay)
else
  dv := Max(Abs(ax), Abs(ay));

// Wspolzedne piksela
if dv <> 1 then
  begin
  ax := ax / dv;
  ay := ay / dv;
  end;              }

Result.X := Round(ax*(FWidth*FSamples-1));
Result.Y := Round(ay*(FHeight*FSamples-1));

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.PixelsToValue(Left, Top: Integer;
  XMin, XMax, YMin, YMax: Double; var X, Y: Double);
begin
X := (Left / (FWidth-1))*(XMax-XMin)+XMin;
Y := (1-(Top / (FHeight-1)))*(YMax-YMin)+YMin;
end;

// -----------------------------------------------------------------------------
function TMPEGraph.AxesToPixels: TPoint;
const
  TOP_MARGIN = 4;
  BOTTOM_MARGIN = 28;
  LEFT_MARGIN = 64;
  RIGHT_MARGIN = 4;
var
  p: TPoint;
begin

p := ValueToPixels(
  0, 0,
  FAxisXMin, FAxisXMax,
  FAxisYMin, FAxisYMax
  );
 {
Result := Point(
  Min(Max(p.X, LEFT_MARGIN*FSamples), (FWidth-RIGHT_MARGIN)*FSamples-1),
  Max(Min(p.Y, (FHeight-BOTTOM_MARGIN)*FSamples-1), TOP_MARGIN*FSamples)
  );   }
   // TODO: Usunac???
Result := Point(
  StepRange(p.X, 0, FWidth*FSamples-1),
  StepRange(p.Y, 0, FHeight*FSamples-1)
  );

end;

// -----------------------------------------------------------------------------
function TMPEGraph.PointToPixels(DataSeriesIndex, DataPointIndex: Integer): TPoint;
begin

Result := ValueToPixels(
  DataSeries[DataSeriesIndex].DataPoints[DataPointIndex].X, DataSeries[DataSeriesIndex].DataPoints[DataPointIndex].Y,
  FAxisXMin, FAxisXMax,
  FAxisYMin, FAxisYMax
  );

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.FitAxes(Margin: Single = 0);
var
  ds, dp: Integer;
  xmargin, ymargin: Double;
begin

// Zerowanie skali osi
if FAxisXAutomatic then
  begin
  FAxisXMin := 0;
  FAxisXMax := 0;
  FAxisYMin := 0;
  FAxisYMax := 0;
  end;

// Analiza
if DataSeriesCount > 0 then
  for ds := 0 to DataSeriesCount-1 do
    if DataSeries[ds].DataPointsCount > 0 then
      for dp := 0 to DataSeries[ds].DataPointsCount-1 do
        if (ds = 0) and (dp = 0) then
          begin

          // Reset skali osi
          if FAxisXAutomatic then
            begin
            FAxisXMin := DataSeries[0].DataPoints[0].X;
            FAxisXMax := DataSeries[0].DataPoints[0].X;
            end;

          if FAxisYAutomatic then
            begin
            FAxisYMin := DataSeries[0].DataPoints[0].Y;
            FAxisYMax := DataSeries[0].DataPoints[0].Y;
            end;

          end
        else
          begin

          // X
          if FAxisXAutomatic then
            if DataSeries[ds].DataPoints[dp].X < FAxisXMin then
              FAxisXMin := DataSeries[ds].DataPoints[dp].X
            else if DataSeries[ds].DataPoints[dp].X > FAxisXMax then
              FAxisXMax := DataSeries[ds].DataPoints[dp].X;

          // Y
          if FAxisYAutomatic then
            if DataSeries[ds].DataPoints[dp].Y < FAxisYMin then
              FAxisYMin := DataSeries[ds].DataPoints[dp].Y
            else if DataSeries[ds].DataPoints[dp].Y > FAxisYMax then
              FAxisYMax := DataSeries[ds].DataPoints[dp].Y;

          end;

// Pokrywanie sie minimum i maksimum
if FAxisXMin = FAxisXMax then
  begin
  FAxisXMin := FAxisXMin-1;
  FAxisXMax := FAxisXMax+1;
  end;

if FAxisYMin = FAxisYMax then
  begin
  FAxisYMin := FAxisYMin-1;
  FAxisYMax := FAxisYMax+1;
  end;

// Marginesy
if Margin <> 0 then
  begin

  if FAxisXAutomatic then
    begin
    xmargin := (FAxisXMax-FAxisXMin)*Margin;
    FAxisXMin := FAxisXMin-xmargin;
    FAxisXMax := FAxisXMax+xmargin;
    end;

  if FAxisYAutomatic then
    begin
    ymargin := (FAxisYMax-FAxisYMin)*Margin;
    FAxisYMin := FAxisYMin-ymargin;
    FAxisYMax := FAxisYMax+ymargin;
    end;

  end;

end;



// -----------------------------------------------------------------------------
procedure TMPEGraph.PixelsToPoint(Left, Top: Integer; var X, Y: Double);
begin

PixelsToValue(
  Left, Top,
  FAxisXMin, FAxisXMax,
  FAxisYMin, FAxisYMax,
  X, Y
  );

end;

// -----------------------------------------------------------------------------
function TMPEGraph.AddDataSeries: TMPEDataSeries;
begin
Result := TMPEDataSeries.Create;
FDataSeries.Add(Result);
end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.AssignDataSeries(DataSeries: TMPEDataSeries);
begin
FDataSeries.Add(DataSeries);
end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.DeleteDataSeries(Index: Integer);
begin
DataSeries[Index].Free;
FDataSeries.Delete(Index);
end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.DeleteAllDataSeries;
var
  c: Integer;
begin

// Zwalnianie serii danych
if FDataSeries.Count > 0 then
  for c := 0 to FDataSeries.Count-1 do
    TMPEDataSeries(FDataSeries[c]).Free;

// Czyszczenie listy
FDataSeries.Clear;

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.Repaint;
var
  i, so, uo: Integer;
  f: Double;
  s: String;
  ds, dp: Integer;
  stime: DWord;
  axes, point, cpoint, npoint: TPoint;
  rect: TRect;
  buffer: TBitmap;
  drawaxes: Boolean;
begin

stime := GetTickCount;

if (FWidth > 0) and (FHeight > 0) then
  begin

  // Automatyczne dopasowanie osi
  FitAxes(0.02); // 0.075

  // Obszar wykresu
  rect := Bounds(0, 0, FWidth*FSamples, FHeight*FSamples);

  // Bufor obrazu
  buffer := TBitmap.Create;
  buffer.Width  := FWidth*FSamples;
  buffer.Height := FHeight*FSamples;

  // Tlo
  buffer.Canvas.Brush.Color := FBackgroundColor;
  buffer.Canvas.FillRect(rect);

  // Osie
  axes := AxesToPixels;

  // Parametry osi
  buffer.Canvas.Brush.Style := bsClear;

  buffer.Canvas.Pen.Color := FAxesColor;
  buffer.Canvas.Pen.Width := Ceil(0.5*2*FSamples);
          
  buffer.Canvas.Font.Color := FFontColor;
  buffer.Canvas.Font.Size := Ceil(8*FSamples);

  // Os odcietych
  buffer.Canvas.MoveTo(0, axes.Y);
  buffer.Canvas.LineTo(FWidth*FSamples, axes.Y);

  // Os rzednych
  buffer.Canvas.MoveTo(axes.X, 0);
  buffer.Canvas.LineTo(axes.X, FHeight*FSamples);

  // Wartosci osi odcietych
  if axes.Y+28*FSamples <= FHeight*FSamples-1 then
    begin
    so := 1;
    uo := 1;
    end
  else
    begin
    so := -1;
    uo := 0;
    end;

  f := (FAxisXMax-FAxisXMin) / (FWidth/50);
  for i := Round(FAxisXMin / f) to Round(FAxisXMax / f) do
    begin
            
    point := ValueToPixels(i*f, 0, FAxisXMin, FAxisXMax, FAxisYMin, FAxisYMax);

    if (point.X >= 0) and (point.X <= FWidth*FSamples-1) then
      begin
      buffer.Canvas.MoveTo(point.X, axes.Y);
      buffer.Canvas.LineTo(point.X, axes.Y+so*8*FSamples);
      s := FloatToStrF(i*f, ffGeneral, 3, 2);
      buffer.Canvas.TextOut(point.X-(buffer.Canvas.TextWidth(s) div 2), axes.Y+so*((1-uo)*buffer.Canvas.TextHeight(s)+10*FSamples), s);
      end;

    end;

  // Wartosci osi rzednych
  if axes.X-64*FSamples >= 0 then
    begin
    so := -1;
    uo := 0;
    end
  else
    begin
    so := 1;
    uo := 1;
    end;

  f := (FAxisYMax-FAxisYMin) / (FHeight/50);
  for i := Round(FAxisYMin / f) to Round(FAxisYMax / f) do
    begin
            
    point := ValueToPixels(0, i*f, FAxisXMin, FAxisXMax, FAxisYMin, FAxisYMax);

    if (point.Y >= 0) and (point.Y <= FHeight*FSamples-1) then
      begin
      buffer.Canvas.MoveTo(axes.X, point.Y);
      buffer.Canvas.LineTo(axes.X+so*8*FSamples, point.Y);
      s := FloatToStrF(i*f, ffGeneral, 3, 2);
      buffer.Canvas.TextOut(axes.X+so*((1-uo)*buffer.Canvas.TextWidth(s)+12*FSamples), point.Y-(buffer.Canvas.TextHeight(s) div 2), s);
      end;

    end;

  // Serie danych
  if DataSeriesCount > 0 then
    for ds := 0 to DataSeriesCount-1 do
      if (FAxisXMin < FAxisXMax) and
         (FAxisYMin < FAxisYMax) then
        begin

        // Dane punktow
        if DataSeries[ds].DataPointsCount > 1 then
          for dp := 0 to DataSeries[ds].DataPointsCount-2 do
            begin

            // Punkty
            cpoint := PointToPixels(ds, dp);
            npoint := PointToPixels(ds, dp+1);

            // Wykres
            if (Max(0, Min(cpoint.X, npoint.X)) <= Min(rect.Right, Max(cpoint.X, npoint.X))) and
               (Max(0, Min(cpoint.Y, npoint.Y)) <= Min(rect.Bottom, Max(cpoint.Y, npoint.Y))) then
              begin
              buffer.Canvas.Pen.Color := DataSeries[ds].Color;
              buffer.Canvas.Pen.Width := Ceil(0.5*DataSeries[ds].LineWidth*FSamples);
              buffer.Canvas.MoveTo(cpoint.X, cpoint.Y);
              buffer.Canvas.LineTo(npoint.X, npoint.Y);
              end;

            end;

        end;

  // Nazwy serii danych
  if DataSeriesCount > 0 then
    begin

    // Parametry czcionki
    buffer.Canvas.Font.Size := Ceil(8*FSamples);

    // Margines gorny
    i := 4*FSamples;

    // Serie danych
    for ds := 0 to DataSeriesCount-1 do
      begin
      buffer.Canvas.Font.Color := DataSeries[ds].Color;
      buffer.Canvas.TextOut(buffer.Width - (buffer.Canvas.TextWidth(DataSeries[ds].Name)+8*FSamples), i, DataSeries[ds].Name);
      Inc(i, buffer.Canvas.TextHeight(DataSeries[ds].Name) + 4*FSamples);
      end;

    end;

  // Rzutowanie bufora
  if FSamples = 1 then
    FBitmap.Canvas.CopyRect(Bounds(0,0,FWidth,FHeight), buffer.Canvas, Bounds(0,0,FWidth,FHeight))
  else
    Resample.Strecth(buffer, FBitmap, ResampleFilters[2].Filter, ResampleFilters[2].Width);

  // Zwalnianie
  buffer.Free;

  end;

// Odswierzenie
if Assigned(FImage) then
  FImage.Picture.Bitmap := FBitmap;

// Czas operacji
FRepaintTime := GetTickCount - stime;

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.ExportToFile(const Filename: String);
var
  f: TextFile;
  ds, dp, maxpoints: Integer;
  s: String;
begin

// Szukanie maksymalnej ilosci danych w serii
maxpoints := 0;

if DataSeriesCount > 0 then
  for ds := 0 to DataSeriesCount-1 do
    if DataSeries[ds].DataPointsCount > maxpoints then
      maxpoints := DataSeries[ds].DataPointsCount;

// Otwarcie pliku
AssignFile(f, FileName);
ReWrite(f);

// Naglowki serii danych
s := '';

if DataSeriesCount > 0 then
  for ds := 0 to DataSeriesCount-1 do
    begin
    if ds > 0 then s := s + #9;
    s := s +
      OUTPUT_DATA_COLUMNS[DataSeries[ds].AxisXColumn].Name +
      ' [' + OUTPUT_DATA_COLUMNS[DataSeries[ds].AxisXColumn].Metric + ']' + #9 +
      OUTPUT_DATA_COLUMNS[DataSeries[ds].AxisYColumn].Name +
      ' [' + OUTPUT_DATA_COLUMNS[DataSeries[ds].AxisYColumn].Metric + ']';
    end;

WriteLn(f, s);

// Zapis danych
if maxpoints > 0 then
  for dp := 0 to maxpoints-1 do
    begin
    
    s := '';

    for ds := 0 to DataSeriesCount-1 do
      if dp <= DataSeries[ds].DataPointsCount-1 then
        begin
        if ds > 0 then s := s + #9;
        s := s +
          FloatToStrF(DataSeries[ds].DataPoints[dp].X, ffGeneral, 16, 2) + #9 +
          FloatToStrF(DataSeries[ds].DataPoints[dp].Y, ffGeneral, 16, 2);
        end;

    WriteLn(f, s);

    end;

CloseFile(f);

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.LoadFromStream(Stream: TStream);
var
  buffer: TMPEBuffer;
  i, u, dscount, dpcount: Integer;
  dataseries: TMPEDataSeries;
begin

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Dane wykresu
FName := buffer.ReadString;
FAxesColor := buffer.ReadInteger;
FBackgroundColor := buffer.ReadInteger;
FFontColor := buffer.ReadInteger;
FAxisXAutomatic := buffer.ReadBoolean;
FAxisXMin := buffer.ReadDouble;
FAxisXMax := buffer.ReadDouble;
FAxisYAutomatic := buffer.ReadBoolean;
FAxisYMin := buffer.ReadDouble;
FAxisYMax := buffer.ReadDouble;

// Dane serii danych
dscount := buffer.ReadInteger;

if dscount > 0 then
  for i := 0 to dscount-1 do
    begin
    dataseries := AddDataSeries;
    dataseries.Name := buffer.ReadString;
    dataseries.Color := buffer.ReadInteger;
    dataseries.LineWidth := buffer.ReadByte;
    dataseries.AxisXColumn := buffer.ReadByte;
    dataseries.AxisYColumn := buffer.ReadByte;

    // Dane punktow
    dpcount := buffer.ReadInteger;

    if dpcount > 0 then
      for u := 0 to dpcount-1 do
        dataseries.AddDataPoint(buffer.ReadDouble, buffer.ReadDouble);

    end;

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEGraph.SaveToStream(Stream: TStream);
var
  buffer: TMPEBuffer;
  i, u: Integer;
begin

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Dane wykresu
buffer.WriteData(FName);
buffer.WriteData(FAxesColor);
buffer.WriteData(FBackgroundColor);
buffer.WriteData(FFontColor);
buffer.WriteData(FAxisXAutomatic);
buffer.WriteData(FAxisXMin);
buffer.WriteData(FAxisXMax);
buffer.WriteData(FAxisYAutomatic);
buffer.WriteData(FAxisYMin);
buffer.WriteData(FAxisYMax);

// Dane serii danych
buffer.WriteData(DataSeriesCount);

if DataSeriesCount > 0 then
  for i := 0 to DataSeriesCount-1 do
    begin
    buffer.WriteData(DataSeries[i].Name);
    buffer.WriteData(DataSeries[i].Color);
    buffer.WriteData(DataSeries[i].LineWidth);
    buffer.WriteData(DataSeries[i].AxisXColumn);
    buffer.WriteData(DataSeries[i].AxisYColumn);

    // Dane punktow
    buffer.WriteData(DataSeries[i].DataPointsCount);

    if DataSeries[i].DataPointsCount > 0 then
      for u := 0 to DataSeries[i].DataPointsCount-1 do
        begin
        buffer.WriteData(DataSeries[i].DataPoints[u].X);
        buffer.WriteData(DataSeries[i].DataPoints[u].Y);
        end;

    end;

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEGraph.Create;
begin

inherited;

// Parametry
FName := '';
FAxesColor := $A0A0A0;
FBackgroundColor := $F8F8F8;
FFontColor := $000000;
FRepaintTime := 0;
FSamples := 1;
FWidth := 0;
FHeight := 0;

FAxisXAutomatic := False;
FAxisXMin := -1;
FAxisXMax := 1;
FAxisYAutomatic := False;
FAxisYMin := -1;
FAxisYMax := 1;

FDataSeries := TList.Create;

FBitmap := TBitmap.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEGraph.Destroy;
begin

// Zwalnianie
DeleteAllDataSeries;

// Parametry
FDataSeries.Free;

FBitmap.Free;

inherited;

end;

end.
