unit MPEGraphEdit;

interface

// -----------------------------------------------------------------------------
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, ExtCtrls,
  Math,
  MPEExt, MPEGraph, MPEProject;

// -----------------------------------------------------------------------------
type
  TGraphTheme = record
    Name: String;
    BackgroundColor: Integer;
    AxesColor: Integer;
    FontColor: Integer;
  end;

// -----------------------------------------------------------------------------
const
  GRAPH_THEMES_COUNT = 5;
  GRAPH_THEMES: array[0..GRAPH_THEMES_COUNT-1] of TGraphTheme = (
    (Name: 'Default'; BackgroundColor: $F8F8F8; AxesColor: $A0A0A0; FontColor: $000000),
    (Name: 'Dark'; BackgroundColor: $000000; AxesColor: $00FFFF; FontColor: $FFFFFF),
    (Name: 'Dolores'; BackgroundColor: $E4FAFF; AxesColor: $25A8D0; FontColor: $7C6240),
    (Name: 'Ocean Sunset'; BackgroundColor: $7A9B9C; AxesColor: $525940; FontColor: $93D3FF),
    (Name: 'Pistachio'; BackgroundColor: $527E67; AxesColor: $79CAB7; FontColor: $B1E8F6)
    );
  DATA_SERIES_COLORS_COUNT = 10;
  DATA_SERIES_COLORS: array[0..DATA_SERIES_COLORS_COUNT-1] of TColor = (
    clBlue, clRed, clGreen, clNavy, clMaroon, clFuchsia, clAqua, clPurple,
    clTeal, clOlive
    );

// -----------------------------------------------------------------------------
type
  TGraphEditForm = class(TForm)
    StatusBar: TStatusBar;
    PageControl: TPageControl;
    TabSheet1: TTabSheet;
    TabSheet2: TTabSheet;
    OKBtn: TButton;
    GroupBox3: TGroupBox;
    Label3: TLabel;
    GraphNameEdit: TEdit;
    TabSheet3: TTabSheet;
    GroupBox4: TGroupBox;
    Label5: TLabel;
    XAxisMinEdit: TEdit;
    XAxisAutoCheckBox: TCheckBox;
    Label6: TLabel;
    XAxisMaxEdit: TEdit;
    GroupBox5: TGroupBox;
    Label7: TLabel;
    Label8: TLabel;
    YAxisMinEdit: TEdit;
    YAxisAutoCheckBox: TCheckBox;
    YAxisMaxEdit: TEdit;
    GroupBox6: TGroupBox;
    Label9: TLabel;
    GraphBackgroundClColorBox: TColorBox;
    Label10: TLabel;
    GraphAxesClColorBox: TColorBox;
    GraphFontClColorBox: TColorBox;
    Label11: TLabel;
    TabSheet4: TTabSheet;
    GroupBox1: TGroupBox;
    DataSeriesListView: TListView;
    DeleteDataSeriesBtn: TButton;
    GroupBox2: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    Label4: TLabel;
    DataSeriesXAxisComboBox: TComboBox;
    AddDataSeriesBtn: TButton;
    DataSeriesYAxisComboBox: TComboBox;
    DataSeriesNameEdit: TEdit;
    Label13: TLabel;
    DataSeriesLineClColorBox: TColorBox;
    Label14: TLabel;
    DataSeriesLineWidthComboBox: TComboBox;
    Label15: TLabel;
    CancelBtn: TButton;
    GroupBox7: TGroupBox;
    Label12: TLabel;
    GraphThemeComboBox: TComboBox;
    GraphThemeApplyBtn: TButton;
    DataSeriesAutoNameCheckBox: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure AddDataSeriesBtnClick(Sender: TObject);
    procedure DeleteDataSeriesBtnClick(Sender: TObject);
    procedure XAxisAutoCheckBoxClick(Sender: TObject);
    procedure YAxisAutoCheckBoxClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure GraphThemeApplyBtnClick(Sender: TObject);
    procedure DataSeriesAutoNameCheckBoxClick(Sender: TObject);
    procedure DataSeriesAxisComboBoxChange(Sender: TObject);
  private
  public
    // Metody
    procedure AddDataSeries(const Name: String;
      AxisXColumn, AxisYColumn: Byte; Color: Integer; LineWidth: Byte);

    procedure LoadFromProject(Index: Integer);
    procedure SaveToProject(Index: Integer = -1);

  end;

// -----------------------------------------------------------------------------
var
  GraphEditForm: TGraphEditForm;

implementation

{$R *.dfm}

// -----------------------------------------------------------------------------
procedure TGraphEditForm.AddDataSeries(const Name: String;
  AxisXColumn, AxisYColumn: Byte; Color: Integer; LineWidth: Byte);
var
  item: TListItem;
  ds: TMPEDataSeries;
begin

// Tworzenie serii danych
ds := TMPEDataSeries.Create;
ds.Name := Name;
ds.AxisXColumn := AxisXColumn;
ds.AxisYColumn := AxisYColumn;
ds.Color := Color;
ds.LineWidth := LineWidth;

// Dodawanie pozycji do listy
item := DataSeriesListView.Items.Add;
item.Data := ds;
item.Caption := Name;
item.SubItems.Add(GetColumnFullName(OUTPUT_DATA_COLUMNS, AxisXColumn));
item.SubItems.Add(GetColumnFullName(OUTPUT_DATA_COLUMNS, AxisYColumn));
item.SubItems.Add('#'+ IntToHex(Color, 6));
item.SubItems.Add(IntToStr(LineWidth));

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.LoadFromProject(Index: Integer);
var
  i: Integer;
  graph: TMPEGraph;
  ds: TMPEDataSeries;
begin

// Parametry wykresu
graph := Project.Graphs.Graphs[Index];

GraphNameEdit.Text := graph.Name;

GraphAxesClColorBox.Selected := graph.AxesColor;
GraphBackgroundClColorBox.Selected := graph.BackgroundColor;
GraphFontClColorBox.Selected := graph.FontColor;

XAxisAutoCheckBox.Checked := graph.AxisXAutomatic;
XAxisMinEdit.Text := FloatToStrF(graph.AxisXMin, ffGeneral, 3, 2);
XAxisMaxEdit.Text := FloatToStrF(graph.AxisXMax, ffGeneral, 3, 2);

YAxisAutoCheckBox.Checked := graph.AxisYAutomatic;
YAxisMinEdit.Text := FloatToStrF(graph.AxisYMin, ffGeneral, 3, 2);
YAxisMaxEdit.Text := FloatToStrF(graph.AxisYMax, ffGeneral, 3, 2);

// Serie danych
if graph.DataSeriesCount > 0 then
  for i := 0 to graph.DataSeriesCount-1 do
    begin
    ds := graph.DataSeries[i];
    AddDataSeries(ds.Name, ds.AxisXColumn, ds.AxisYColumn, ds.Color, ds.LineWidth);
    end;

// Interface
DataSeriesLineClColorBox.Selected := DATA_SERIES_COLORS[Min(DataSeriesListView.Items.Count, DATA_SERIES_COLORS_COUNT-1)];

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.SaveToProject(Index: Integer = -1);
var
  i: Integer;
  graph: TMPEGraph;
  ds: TMPEDataSeries;
begin

// Dodanie/edycja
if Index = -1 then
  graph := Project.Graphs.AddGraph
else
  graph := Project.Graphs.Graphs[Index];

// Parametry wykresu
graph.Name := GraphNameEdit.Text;

graph.AxesColor := GraphAxesClColorBox.Selected;
graph.BackgroundColor := GraphBackgroundClColorBox.Selected;
graph.FontColor := GraphFontClColorBox.Selected;

graph.AxisXAutomatic := XAxisAutoCheckBox.Checked;
graph.AxisXMin := StrToFloat(XAxisMinEdit.Text);
graph.AxisXMax := StrToFloat(XAxisMaxEdit.Text);

graph.AxisYAutomatic := YAxisAutoCheckBox.Checked;
graph.AxisYMin := StrToFloat(YAxisMinEdit.Text);
graph.AxisYMax := StrToFloat(YAxisMaxEdit.Text);

// Serie danych
graph.DeleteAllDataSeries;

if DataSeriesListView.Items.Count > 0 then
  for i := 0 to DataSeriesListView.Items.Count-1 do
    begin
    ds := TMPEDataSeries(DataSeriesListView.Items[i].Data);
    ds.ImportFromOutputData(Project.OutputData);
    graph.AssignDataSeries(ds);
    end;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.XAxisAutoCheckBoxClick(Sender: TObject);
begin
XAxisMinEdit.Enabled := not XAxisAutoCheckBox.Checked;
XAxisMaxEdit.Enabled := not XAxisAutoCheckBox.Checked;
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.YAxisAutoCheckBoxClick(Sender: TObject);
begin
YAxisMinEdit.Enabled := not YAxisAutoCheckBox.Checked;
YAxisMaxEdit.Enabled := not YAxisAutoCheckBox.Checked;
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.AddDataSeriesBtnClick(Sender: TObject);
begin

// Tworzenie serii danych
AddDataSeries(
  DataSeriesNameEdit.Text,
  DataSeriesXAxisComboBox.ItemIndex,
  DataSeriesYAxisComboBox.ItemIndex,
  DataSeriesLineClColorBox.Selected,
  StrToInt(DataSeriesLineWidthComboBox.Text)
  );

// Resetowanie pol edycji
DataSeriesNameEdit.Text := 'new data series';
DataSeriesXAxisComboBox.ItemIndex := 2;
DataSeriesYAxisComboBox.ItemIndex := DataSeriesYAxisComboBox.ItemIndex+1;
DataSeriesAxisComboBoxChange(nil);
DataSeriesLineClColorBox.Selected := DATA_SERIES_COLORS[Min(DataSeriesListView.Items.Count, DATA_SERIES_COLORS_COUNT-1)];
DataSeriesLineWidthComboBox.ItemIndex := DataSeriesLineWidthComboBox.Tag;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DataSeriesAutoNameCheckBoxClick(Sender: TObject);
begin
DataSeriesNameEdit.Enabled := not DataSeriesAutoNameCheckBox.Checked;
DataSeriesAxisComboBoxChange(nil);
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DataSeriesAxisComboBoxChange(Sender: TObject);
begin

if DataSeriesAutoNameCheckBox.Checked then
  DataSeriesNameEdit.Text :=
    GetColumnFullName(OUTPUT_DATA_COLUMNS, DataSeriesYAxisComboBox.ItemIndex) + ' = f (' +
    GetColumnFullName(OUTPUT_DATA_COLUMNS, DataSeriesXAxisComboBox.ItemIndex) + ')';

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DeleteDataSeriesBtnClick(Sender: TObject);
var
  index: Integer;
begin

index := DataSeriesListView.ItemIndex;

if index <> -1 then
  begin
  TMPEDataSeries(DataSeriesListView.Items[index].Data).Free;
  DataSeriesListView.Items.Delete(index);
  DataSeriesListView.ItemIndex := -1;
  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  i: Integer;
begin

// Zwalnianie nie zatwierdzonych serii danych
if ModalResult <> mrOK then
  if DataSeriesListView.Items.Count > 0 then
    for i := 0 to DataSeriesListView.Items.Count-1 do
      TMPEDataSeries(DataSeriesListView.Items[i].Data).Free;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.FormCreate(Sender: TObject);
var
  b: Byte;
  s: String;
begin

// Aktywna zakladka
PageControl.ActivePageIndex := 0;

// Kolumny pliku wyjsciowego
for b := 0 to OUTPUT_DATA_COLUMNS_COUNT-1 do
  begin
  s := GetColumnFullName(OUTPUT_DATA_COLUMNS, b);
  DataSeriesXAxisComboBox.Items.Add(s);
  DataSeriesYAxisComboBox.Items.Add(s);
  end;

// Interface
DataSeriesXAxisComboBox.ItemIndex := 2;
DataSeriesYAxisComboBox.ItemIndex := 6;
DataSeriesAxisComboBoxChange(nil);

// Schematy kolorow
for b := 0 to GRAPH_THEMES_COUNT-1 do
  GraphThemeComboBox.Items.Add(GRAPH_THEMES[b].Name);

GraphThemeComboBox.ItemIndex := 0;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.GraphThemeApplyBtnClick(Sender: TObject);
var
  i: Integer;
begin

i := GraphThemeComboBox.ItemIndex;

if i <> -1 then
  begin
  GraphBackgroundClColorBox.Selected := GRAPH_THEMES[i].BackgroundColor;
  GraphAxesClColorBox.Selected := GRAPH_THEMES[i].AxesColor;
  GraphFontClColorBox.Selected := GRAPH_THEMES[i].FontColor;
  end;

end;

end.
