unit FastFourier;

      {*********************************************************}
      {***   FAST FOURIER TRANSFORM   Component              ***}
      {***   Autor : Natalia Walkowicz, grudzien 2003        ***}
      {***   Maxymalna liczba probek: N = 2 147 483 648      ***}
      {*********************************************************}

interface

uses
  Windows, Messages, SysUtils, Classes, Math, Complex;

type
     TFastFourier = class(TComponent)
     private
          FNumSamples    : LongWord;
          function  IsPowerOfTwo(x : LongWord) : Boolean;
          function  NumberOfBitsNeeded(Ile : LongWord) : LongWord;
          function  ReverseBits(index, NumBits : LongWord) : LongWord;
          procedure FourierTransform(Angle : Extended);
          procedure SetNumSamples(Value : LongWord);
     public
          InBuffer  : array of TComplex;
          OutBuffer : array of TComplex;
          procedure FFT;
          procedure IFFT;
          function  NumberOfSamples(Count : LongWord) : LongWord;
          function  Real(idx : LongWord) : Extended;
          function  Imag(idx : LongWord) : Extended;
          function  Absol(idx : LongWord) : Extended;
          function  Energy(idx, SamplFreq : LongWord) : Extended;
          function  Power(idx : LongWord) : Extended;
          function  Phase(idx : LongWord) : Extended;
          function  Frequency(idx, SamplFreq : LongWord) : Extended;
          procedure Free;
          constructor Create(AOwner : TComponent); override;
          destructor  Destroy; override;
     published
          property NumSamples : LongWord read FNumSamples write SetNumSamples;
end;

procedure Register;

implementation

procedure Register;
begin
     RegisterComponents('Engineering', [TFastFourier]);
end;

constructor TFastFourier.Create(AOwner : TComponent);
begin
     inherited Create(AOwner);
     NumSamples:=0;
end;

destructor TFastFourier.Destroy;
begin
     SetLength(InBuffer,0);
     SetLength(OutBuffer,0);
     inherited;
end;

procedure TFastFourier.SetNumSamples(Value : LongWord);
begin
     FNumSamples:=Value;
     SetLength(InBuffer,Value);
     SetLength(OutBuffer,Value);
end;

function TFastFourier.NumberOfSamples(Count: LongWord): LongWord;
var   i:  LongWord;
begin
     if IsPowerOfTwo(Count) then
        Result:=Count
     else
     begin
          Result := 1;
          i:=0;
          while ((i<=31) and (Result<Count)) do
          begin
               Result := Result SHL 1;
               i:=i+1;
          end;
     end;
end;

function TFastFourier.IsPowerOfTwo(x : LongWord) : Boolean;
var   i, y :  LongWord;
begin
     y:=2;
     for i:=1 to 31 do
     begin
          if x = y then
          begin
               Result := True;
               exit;
          end;
          y:=y SHL 1;
     end;

     Result := False;
end;

function TFastFourier.NumberOfBitsNeeded(Ile : LongWord) : LongWord;
var i : LongWord;
begin
     Result := 0;
     for i:=0 to 31 do
     begin
          if (Ile AND (1 SHL i)) <> 0 then
          begin
               Result:=i;
               Exit;
          end;
     end;
end;

function TFastFourier.ReverseBits(index, NumBits : LongWord) : LongWord;
var i, rev : LongWord;
begin
     rev:=0;
     for i:=0 to NumBits-1 do
     begin
          rev:=(rev SHL 1) OR (index AND 1);
          index:=index SHR 1;
     end;

     Result:=rev;
end;

procedure TFastFourier.FourierTransform(Angle : Extended);
var NumBits, i, j, k, n, BlockSize, BlockEnd : LongWord;
    delta_angle, delta_ar                    : Extended;
    alpha, beta                              : Extended;
    tr, ti, ar, ai                           : Extended;
begin
     NumBits:=NumberOfBitsNeeded(FNumSamples);

     for i:=0 to FNumSamples-1 do
     begin
          j:=ReverseBits(i, NumBits);
          OutBuffer[j]:=InBuffer[i];
     end;

     BlockEnd:=1;
     BlockSize:=2;
     while BlockSize<=FNumSamples do
     begin
          delta_angle:=Angle/BlockSize;
          alpha:=sin(0.5* delta_angle);
          alpha:=2.0*alpha*alpha;
          beta:=sin(delta_angle);

          i:=0;
          while i<FNumSamples do
          begin
               ar:=1.0;    // cos(0)
               ai:=0.0;    // sin(0)

               j:=i;
               for n:=0 to BlockEnd-1 do
               begin
                    k:=j+BlockEnd;
                    tr:=ar*OutBuffer[k].Re-ai*OutBuffer[k].Im;
                    ti:=ar*OutBuffer[k].Im+ai*OutBuffer[k].Re;
                    OutBuffer[k].Re:=OutBuffer[j].Re-tr;
                    OutBuffer[k].Im:=OutBuffer[j].Im-ti;
                    OutBuffer[j].Re:=OutBuffer[j].Re+tr;
                    OutBuffer[j].Im:=OutBuffer[j].Im+ti;
                    delta_ar:=alpha*ar+beta*ai;
                    ai:=ai-(alpha*ai-beta*ar);
                    ar:=ar-delta_ar;
                    INC(j);
               end;
               i:=i+BlockSize;
          end;
          BlockEnd:=BlockSize;
          BlockSize:=BlockSize SHL 1;
     end;
end;

procedure TFastFourier.FFT;
begin
     FourierTransform(2*PI);
end;

procedure TFastFourier.IFFT;
var i: LongWord;
begin
     FourierTransform(-2*PI);
     for i:=0 to FNumSamples-1 do
     begin
          OutBuffer[i].Re:=OutBuffer[i].Re/FNumSamples;
          OutBuffer[i].Im:=OutBuffer[i].Im/FNumSamples;
     end;
end;

function TFastFourier.Imag(idx: LongWord): Extended;
var z : TComplex;
begin
     z:=OutBuffer[idx];
     Result:=z.Im;
end;

function TFastFourier.Real(idx: LongWord): Extended;
var z : TComplex;
begin
     z:=OutBuffer[idx];
     Result:=z.Re;
end;

function TFastFourier.Absol(idx : LongWord) : Extended;
var z : TComplex;
begin
     z:=OutBuffer[idx];
     if idx=0 then
        Result:=sqrt(z.Re*z.Re+z.Im*z.Im)/FNumSamples
     else if idx>0 then
        Result:=sqrt(z.Re*z.Re+z.Im*z.Im)/FNumSamples*2
     else
        Result:=0;
end;

function TFastFourier.Power(idx: LongWord): Extended;
var z : TComplex;
begin
     z:=OutBuffer[idx];
     if idx=0 then
        Result:=(z.Re*z.Re+z.Im*z.Im)/FNumSamples
     else if idx>0 then
        Result:=(z.Re*z.Re+z.Im*z.Im)/FNumSamples*2
     else
        Result:=0;
end;

function TFastFourier.Energy(idx, SamplFreq : LongWord): Extended;
var x : Extended;
begin
     x:=Power(idx);
     Result:=x*x/FNumSamples*SamplFreq;
end;

function TFastFourier.Phase(idx: LongWord): Extended;
var re, im : Extended;
begin
     re:=Real(idx);
     im:=Imag(idx);
     if abs(re)>1e-5 then
        Result:=ArcTan(im/re)
     else
        Result:=0;
end;

function TFastFourier.Frequency(idx, SamplFreq: LongWord): Extended;
begin
     if idx<FNumSamples then
        Result:=SamplFreq/FNumSamples*idx
     else
        Result:=0;
end;

procedure TFastFourier.Free;
begin
     SetLength(InBuffer,0);
     SetLength(OutBuffer,0);
end;

end.
