unit MPEAnimations;

interface

// -----------------------------------------------------------------------------
uses
  Windows, SysUtils, Classes, Graphics,
  Math, Resample, GIFImage,
  MPEBitmap, MPEExt;

// -----------------------------------------------------------------------------
type
  TBitmapChannel = (bcA, bcR, bcG, bcB, bcRGB, bcARGB);
  
  TMPEAnimations = class(TObject)
  private
    // Parametry
    FPosition: Integer;
    FWidth: Integer;
    FHeight: Integer;
    
    FFrames: array[0..1, 0..2] of TList;

    // Metody parametrow
    function GetCount: Integer;
    procedure SetPosition(Position: Integer);
    function GetFrame(Slice: Byte; Axis: Byte; Index: Integer): TMPEBitmap;

  public
    // Parametry
    property Count: Integer read GetCount;
    property Position: Integer read FPosition write SetPosition;
    property Width: Integer read FWidth write FWidth;
    property Height: Integer read FHeight write FHeight;

    property Frames[Slice: Byte; Axis: Byte; Index: Integer]: TMPEBitmap read GetFrame;

    // Metody
    function AddFrame(Slice: Byte; Axis: Byte): TMPEBitmap;
    procedure DeleteFrame(Slice: Byte; Axis: Byte; Index: Integer);
    procedure DeleteAllFrames;

    procedure ImpotFromFile(Slice: Byte; Axis: Byte; const FileName: String);
    procedure ImportFromDirectory(const Directory: String);
    procedure ImportFromProjectDirectory;

    procedure ExportToFile(Slice: Byte; Axis: Byte; Delay: Word; const FileName: String);

    procedure Draw(Slice: Byte; Axis: Byte; Canvas: TCanvas; X, Y: Integer);
    procedure DrawSlice(Slice: Byte; Canvas: TCanvas; Rect: TRect);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;

  end;

implementation

// -----------------------------------------------------------------------------
uses
  MPEProject;

// -----------------------------------------------------------------------------
function TMPEAnimations.GetCount: Integer;
begin
Result := Min(Min(FFrames[0,0].Count, FFrames[0,1].Count), FFrames[0,2].Count);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.SetPosition(Position: Integer);
begin

if FFrames[0,0].Count > 0 then
  FPosition := BoundsRange(Position, 0, Count-1)
else
  FPosition := -1;

end;

// -----------------------------------------------------------------------------
function TMPEAnimations.GetFrame(Slice: Byte; Axis: Byte; Index: Integer): TMPEBitmap;
begin
Result := TMPEBitmap(FFrames[Slice, Axis][Index]);
end;

// -----------------------------------------------------------------------------
function TMPEAnimations.AddFrame(Slice: Byte; Axis: Byte): TMPEBitmap;
begin
Result := TMPEBitmap.Create;
FFrames[Slice, Axis].Add(Result);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.DeleteFrame(Slice: Byte; Axis: Byte; Index: Integer);
begin
TMPEBitmap(FFrames[Slice, Axis][Index]).Free;
FFrames[Slice, Axis].Delete(Index);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.DeleteAllFrames;
var
  s, b: Byte;
  i: Integer;
begin

// Zwalnianie klatek
for s := 0 to Length(FFrames)-1 do
  for b := 0 to Length(FFrames[s])-1 do
    begin

    if FFrames[s,b].Count > 0 then
      for i := 0 to FFrames[s,b].Count-1 do
        TMPEBitmap(FFrames[s,b][i]).Free;

    FFrames[s,b].Clear;

    end;

// Pozycja
FPosition := -1;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ImpotFromFile(Slice: Byte; Axis: Byte;
  const FileName: String);
var
  buffer: array[0..1] of TMPEBitmap;
  frame, dst_buffer: TMPEBitmap;
  factor: Single;
begin

// Tworzenie buforow
buffer[0] := TMPEBitmap.Create;
buffer[1] := TMPEBitmap.Create;

// Ladowanie pliku
buffer[0].LoadFromFileEx(FileName);

// Dodawanie klatki animacji
frame := AddFrame(Slice, Axis);
frame.Canvas.Brush.Color := clWhite;
frame.Width := FWidth;
frame.Height := FWidth;

// Przeskalowanie
if (buffer[0].Width > FWidth) or (buffer[0].Height > FHeight) then
  begin

  // Wymiary bufora docelowego
  if buffer[0].Width-FWidth > buffer[0].Height-FHeight then
    factor :=  FWidth / buffer[0].Width
  else
    factor :=  FHeight / buffer[0].Height;

  buffer[1].Width := Round(buffer[0].Width*factor);
  buffer[1].Height := Round(buffer[0].Height*factor);

  // Przeskalowanie
  Strecth(buffer[0], buffer[1], ResampleFilters[5].Filter, ResampleFilters[5].Width);

  dst_buffer := buffer[1];
  
  end
else
  dst_buffer := buffer[0];

// Centrowanie
frame.Canvas.CopyRect(
  Bounds((FWidth-dst_buffer.Width) div 2, (FHeight-dst_buffer.Height) div 2, dst_buffer.Width, dst_buffer.Height),
  dst_buffer.Canvas,
  Bounds(0, 0, dst_buffer.Width, dst_buffer.Height)
  );

// Zwalnianie buforow
buffer[0].Free;
buffer[1].Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ImportFromDirectory(const Directory: String);
var
  s, l, i: Integer;
  slicename: String;
  searchrec: TSearchRec;
  files_array: array of String;
  frame_filename_array: array[0..3] of String;
begin

// Czyszczenie klatek
DeleteAllFrames;

// Odczytywanie plikow
if DirectoryExists(Directory) then
  begin
  
  for s := 0 to Length(FFrames)-1 do
    begin

    // Plaszczyzna ciecia
    if s > 0 then
      slicename := '_' + IntToStr(s+1)
    else
      slicename := '';

    // Indeksowanie listy plikow
    SetLength(files_array, 0);
    if FindFirst(Directory + Project.SimName + slicename + '.*.?' + GRAPHICS_OUTPUT_DATA_FILEEXT, faAnyFile, searchrec) = 0 then
      repeat
        if Pos('9999', searchrec.Name) = 0 then
          begin
          l := Length(files_array);
          SetLength(files_array, l+1);
          files_array[l] := searchrec.Name;
          end;
      until FindNext(searchrec) <> 0;
    FindClose(searchrec);

    // Odczytywanie klatek animacji
    if Length(files_array) > 0 then
      begin
      Sort(files_array);
      for i := 0 to Length(files_array)-1 do
        begin
        Split('.', files_array[i], frame_filename_array);
        ImpotFromFile(s, StrToInt(frame_filename_array[2]), Project.Path + files_array[i]);
        end;
      end;

    end;

  end;

// Pozycja animacji
if Count > 0 then
  FPosition := 0
else
  FPosition := -1;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ImportFromProjectDirectory;
begin
ImportFromDirectory(Project.Path);
end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.ExportToFile(Slice: Byte; Axis: Byte; Delay: Word; const FileName: String);
var
  i, l, index: Integer;
  slicename: String;
  searchrec: TSearchRec;
  files_array: array of String;
  frame: TMPEBitmap;
  gif: TGIFImage;
  ext: TGIFGraphicControlExtension;
  loopext: TGIFAppExtNSLoop;
begin

// Odczytywanie plikow
if DirectoryExists(Project.Path) then
  begin

  // Plaszczyzna ciecia
  if Slice > 0 then
    slicename := '_' + IntToStr(Slice+1)
  else
    slicename := '';

  // Indeksowanie listy plikow
  SetLength(files_array, 0);
  if FindFirst(Project.Path + Project.SimName + slicename + '.*.' + IntToStr(Axis) + GRAPHICS_OUTPUT_DATA_FILEEXT, faAnyFile, searchrec) = 0 then
    repeat
      if Pos('9999', searchrec.Name) = 0 then
        begin
        l := Length(files_array);
        SetLength(files_array, l+1);
        files_array[l] := searchrec.Name;
        end;
    until FindNext(searchrec) <> 0;
  FindClose(searchrec);

  // Tworzenie animacji
  if Length(files_array) > 0 then
    begin

    Sort(files_array);

    gif := TGIFImage.Create;
    gif.ColorReduction := rmQuantizeWindows;
    gif.DitherMode := dmFloydSteinberg;
    gif.Compression := gcLZW;

    for i := 0 to Length(files_array)-1 do
      begin

      frame := TMPEBitmap.Create;
      frame.LoadFromFileEx(Project.Path + files_array[i]);

      index := gif.Add(frame);
      if (index = 0) then
        begin
        loopext := TGIFAppExtNSLoop.Create(gif.Images[index]);
        loopext.Loops := 0;
        gif.Images[index].Extensions.Add(loopext);
        end;

      ext := TGIFGraphicControlExtension.Create(gif.Images[index]);
      ext.Delay := Delay div 10;

      ext.Transparent := True;
      ext.TransparentColorIndex := gif.Images[index].Pixels[0, gif.Height-1];

      gif.Images[index].Extensions.Add(ext);

      frame.Free;

      end;

    gif.SaveToFile(FileName);
    gif.Free;

    end;


  end;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.Draw(Slice: Byte; Axis: Byte; Canvas: TCanvas;
  X, Y: Integer);
begin

// Klatka
if (FPosition >= 0) and (FPosition <= FFrames[Slice, Axis].Count-1) then
  Canvas.CopyRect(
    Bounds(X, Y, FWidth, FHeight),
    Frames[Slice, Axis, FPosition].Canvas,
    Bounds(0, 0, FWidth, FHeight)
    )
else
  begin
  Canvas.Pen.Color := clWhite;
  Canvas.Brush.Color := clWhite;
  Canvas.Rectangle(Bounds(X, Y, FWidth, FHeight));
  end;

end;

// -----------------------------------------------------------------------------
procedure TMPEAnimations.DrawSlice(Slice: Byte; Canvas: TCanvas; Rect: TRect);
var
  buffer: TBitmap;
  points: array[0..1] of TPoint;
  hx, hy: Integer;
begin

// Bufor
buffer := TBitmap.Create; // TODO: Utworzyc globalnie i tylko czyscic
buffer.Canvas.Brush.Color := clWhite;
buffer.PixelFormat := pf24bit;
buffer.Width := rect.Right-rect.Left;
buffer.Height := rect.Bottom-rect.Top;

// Wspolzedne punktow
points[0] := Point(Round((buffer.Width*0.5 - FWidth) / 2), Round((buffer.Height*0.5 - FHeight) / 2));
points[1] := Point(Round((buffer.Width*1.5 - FWidth) / 2), Round((buffer.Height*1.5 - FHeight) / 2));
hx := Round(buffer.Width*0.5);
hy := Round(buffer.Height*0.5);

// Plaszczyzny wspolzednych
Draw(Slice, 0, buffer.Canvas, points[1].X, points[1].Y);
Draw(Slice, 1, buffer.Canvas, points[0].X, points[0].Y);
Draw(Slice, 2, buffer.Canvas, points[0].X, points[1].Y);

// Osie
buffer.Canvas.Pen.Color := clGray;
buffer.Canvas.Pen.Style := psDot;
buffer.Canvas.MoveTo(10, hy);
buffer.Canvas.LineTo(buffer.Width-10, hy);
buffer.Canvas.MoveTo(hx, 10);
buffer.Canvas.LineTo(hx, buffer.Height-10);

// Legenda
buffer.Canvas.Font.Color := clGray;
buffer.Canvas.Font.Name := 'Tahoma'; // TODO: Wylaczyc przed "nawias"
buffer.Canvas.TextOut(hx+5+3, hy+5, 'X');
buffer.Canvas.TextOut(hx-15, hy-15, 'Y');
buffer.Canvas.TextOut(hx-15, hy+5, 'Z');

// Zrzut bufora
Canvas.CopyRect(
  rect,
  buffer.Canvas,
  Bounds(0, 0, buffer.Width, buffer.Height)
  );

// Zwolnienie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEAnimations.Create;
var
  s, b: Byte;
begin

inherited Create;

// Parametry
FPosition := -1;
FWidth := 190;
FHeight := 190;

for s := 0 to Length(FFrames)-1 do
  for b := 0 to Length(FFrames[s])-1 do
    FFrames[s, b] := TList.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEAnimations.Destroy;
var
  s, b: Byte;
begin

// Zwalnianie klatek
DeleteAllFrames;

// Parametry
for s := 0 to Length(FFrames)-1 do
  for b := 0 to Length(FFrames[s])-1 do
    FFrames[s, b].Free;

inherited Destroy;

end;

end.
