unit MPEGraphEdit;

interface

// -----------------------------------------------------------------------------
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, StdCtrls, ExtCtrls,
  Math,
  MPEExt, MPEGraph, MPEFFTGraph, MPEWaveletGraph, MPEProject;

// -----------------------------------------------------------------------------
type
  TGraphTheme = record
    Name: String;
    BackgroundColor: Integer;
    AxesColor: Integer;
    FontColor: Integer;
  end;

// -----------------------------------------------------------------------------
const
  GRAPH_THEMES_COUNT = 7;
  GRAPH_THEMES: array[0..GRAPH_THEMES_COUNT-1] of TGraphTheme = (
    (Name: 'Default'; BackgroundColor: $FDFDFD; AxesColor: $C0C0C0; FontColor: $404040),
    (Name: 'Black'; BackgroundColor: $FFFFFF; AxesColor: $000000; FontColor: $000000),
    (Name: 'White'; BackgroundColor: $000000; AxesColor: $FFFFFF; FontColor: $FFFFFF),
    (Name: 'Dark'; BackgroundColor: $000000; AxesColor: $00FFFF; FontColor: $FFFFFF),
    (Name: 'Dolores'; BackgroundColor: $E4FAFF; AxesColor: $25A8D0; FontColor: $7C6240),
    (Name: 'Ocean Sunset'; BackgroundColor: $7A9B9C; AxesColor: $525940; FontColor: $93D3FF),
    (Name: 'Pistachio'; BackgroundColor: $527E67; AxesColor: $79CAB7; FontColor: $B1E8F6)
    );
  DATA_SERIES_COLORS_COUNT = 10;
  DATA_SERIES_COLORS: array[0..DATA_SERIES_COLORS_COUNT-1] of TColor = (
    clBlue, clRed, clGreen, clNavy, clMaroon, clFuchsia, clAqua, clPurple,
    clTeal, clOlive
    );

// -----------------------------------------------------------------------------
type
  TGraphEditForm = class(TForm)
    StatusBar: TStatusBar;
    PageControl: TPageControl;
    GeneralTabSheet: TTabSheet;
    ApperanceTabSheet: TTabSheet;
    OKBtn: TButton;
    GroupBox3: TGroupBox;
    Label3: TLabel;
    GraphNameEdit: TEdit;
    AxesTabSheet: TTabSheet;
    GroupBox4: TGroupBox;
    Label5: TLabel;
    XAxisMinEdit: TEdit;
    XAxisAutoCheckBox: TCheckBox;
    Label6: TLabel;
    XAxisMaxEdit: TEdit;
    GroupBox5: TGroupBox;
    Label7: TLabel;
    Label8: TLabel;
    YAxisMinEdit: TEdit;
    YAxisAutoCheckBox: TCheckBox;
    YAxisMaxEdit: TEdit;
    GroupBox6: TGroupBox;
    GraphBackgroundLabel: TLabel;
    GraphBackgroundClColorBox: TColorBox;
    Label10: TLabel;
    GraphAxesClColorBox: TColorBox;
    GraphFontClColorBox: TColorBox;
    Label11: TLabel;
    DataSeriesTabSheet: TTabSheet;
    GroupBox1: TGroupBox;
    DataSeriesListView: TListView;
    DeleteDataSeriesBtn: TButton;
    GroupBox2: TGroupBox;
    Label1: TLabel;
    Label2: TLabel;
    Label4: TLabel;
    DataSeriesXAxisComboBox: TComboBox;
    AddDataSeriesBtn: TButton;
    DataSeriesYAxisComboBox: TComboBox;
    DataSeriesNameEdit: TEdit;
    Label13: TLabel;
    DataSeriesLineClColorBox: TColorBox;
    Label14: TLabel;
    DataSeriesLineWidthComboBox: TComboBox;
    Label15: TLabel;
    CancelBtn: TButton;
    GroupBox7: TGroupBox;
    Label12: TLabel;
    GraphThemeComboBox: TComboBox;
    GraphThemeApplyBtn: TButton;
    DataSeriesAutoNameCheckBox: TCheckBox;
    DataSeriesFFT: TCheckBox;
    Label16: TLabel;
    GraphTypeComboBox: TComboBox;
    WaveletTabSheet: TTabSheet;
    GroupBox8: TGroupBox;
    WaveletInterpolationLevelTrackBar: TTrackBar;
    Label17: TLabel;
    Label18: TLabel;
    GroupBox9: TGroupBox;
    Label19: TLabel;
    WaveletColorThemeComboBox: TComboBox;
    Label20: TLabel;
    WaveletColorsNumberComboBox: TComboBox;
    GroupBox10: TGroupBox;
    Label21: TLabel;
    Label22: TLabel;
    WaveletSigmaMinEdit: TEdit;
    WaveletSigmaMaxEdit: TEdit;
    GroupBox11: TGroupBox;
    Label23: TLabel;
    WaveletWaveletComboBox: TComboBox;
    WaveletInterpolationLevelLabel: TLabel;
    GroupBox12: TGroupBox;
    GraphShowAxesCheckBox: TCheckBox;
    GraphShowLegendCheckBox: TCheckBox;
    WaveletShowDataSeriesCheckBox: TCheckBox;
    GroupBox13: TGroupBox;
    Label9: TLabel;
    GraphFontSizeComboBox: TComboBox;
    Label24: TLabel;
    procedure FormCreate(Sender: TObject);
    procedure AddDataSeriesBtnClick(Sender: TObject);
    procedure DeleteDataSeriesBtnClick(Sender: TObject);
    procedure XAxisAutoCheckBoxClick(Sender: TObject);
    procedure YAxisAutoCheckBoxClick(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure GraphThemeApplyBtnClick(Sender: TObject);
    procedure DataSeriesAutoNameCheckBoxClick(Sender: TObject);
    procedure DataSeriesAxisComboBoxChange(Sender: TObject);
    procedure DataSeriesFFTClick(Sender: TObject);
    procedure GraphTypeComboBoxChange(Sender: TObject);
    procedure WaveletInterpolationLevelTrackBarChange(Sender: TObject);
  private
  public
    // Metody
    procedure AddDataSeries(const Name: String;
      AxisXColumn, AxisYColumn: Byte; Color: Integer; LineWidth: Byte;
      FFT: Boolean);

    procedure LoadFromProject(Index: Integer);
    procedure SaveToProject(Index: Integer = -1);

  end;

// -----------------------------------------------------------------------------
var
  GraphEditForm: TGraphEditForm;

implementation

{$R *.dfm}

// -----------------------------------------------------------------------------
procedure TGraphEditForm.AddDataSeries(const Name: String;
  AxisXColumn, AxisYColumn: Byte; Color: Integer; LineWidth: Byte;
  FFT: Boolean);
var
  item: TListItem;
  ds: TMPEDataSeries;
begin

// Tworzenie serii danych
if FFT then
  ds := TMPEFFTDataSeries.Create
else
  ds := TMPEDataSeries.Create;
ds.Name := Name;
ds.AxisXColumn := AxisXColumn;
ds.AxisYColumn := AxisYColumn;
ds.Color := Color;
ds.LineWidth := LineWidth;

// Dodawanie pozycji do listy
item := DataSeriesListView.Items.Add;
item.Data := ds;
item.Caption := Name;
item.SubItems.Add(GetColumnFullName(OUTPUT_DATA_COLUMNS, AxisXColumn));
item.SubItems.Add(GetColumnFullName(OUTPUT_DATA_COLUMNS, AxisYColumn));
item.SubItems.Add('#'+ IntToHex(Color, 6));
item.SubItems.Add(IntToStr(LineWidth));

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.LoadFromProject(Index: Integer);
var
  i: Integer;
  graph: TMPEGraph;
  ds: TMPEDataSeries;
begin

// Parametry wykresu
graph := Project.Graphs.Graphs[Index];

GraphNameEdit.Text := graph.Name;
if graph is TMPEWaveletGraph then
  GraphTypeComboBox.ItemIndex := 1
else
  GraphTypeComboBox.ItemIndex := 0;
GraphTypeComboBox.Enabled := False;
GraphTypeComboBoxChange(nil);
GraphShowAxesCheckBox.Checked := graph.ShowAxes;
GraphShowLegendCheckBox.Checked := graph.ShowLegend;

GraphAxesClColorBox.Selected := graph.AxesColor;
GraphBackgroundClColorBox.Selected := graph.BackgroundColor;
GraphFontClColorBox.Selected := graph.FontColor;
GraphFontSizeComboBox.Text := IntToStr(graph.FontSize);

XAxisAutoCheckBox.Checked := graph.AxisXAutomatic;
XAxisMinEdit.Text := FloatToStrF(graph.AxisXMin, ffGeneral, 4, 2);
XAxisMaxEdit.Text := FloatToStrF(graph.AxisXMax, ffGeneral, 4, 2);

YAxisAutoCheckBox.Checked := graph.AxisYAutomatic;
YAxisMinEdit.Text := FloatToStrF(graph.AxisYMin, ffGeneral, 4, 2);
YAxisMaxEdit.Text := FloatToStrF(graph.AxisYMax, ffGeneral, 4, 2);

// Serie danych
if graph.DataSeriesCount > 0 then
  for i := 0 to graph.DataSeriesCount-1 do
    begin
    ds := graph.DataSeries[i];
    AddDataSeries(ds.Name, ds.AxisXColumn, ds.AxisYColumn, ds.Color, ds.LineWidth, ds is TMPEFFTDataSeries);
    end;

// Transformata falkowa
if graph is TMPEWaveletGraph then
  begin
  WaveletWaveletComboBox.ItemIndex := Integer(TMPEWaveletGraph(graph).Wavelet);
  WaveletSigmaMinEdit.Text := FloatToStrF(TMPEWaveletGraph(graph).SigmaMin, ffGeneral, 4, 2);
  WaveletSigmaMaxEdit.Text := FloatToStrF(TMPEWaveletGraph(graph).SigmaMax, ffGeneral, 4, 2);
  WaveletInterpolationLevelTrackBar.Position := Round(TMPEWaveletGraph(graph).InterpolationLevel*10);
  WaveletColorThemeComboBox.ItemIndex := TMPEWaveletGraph(graph).ColorTheme;
  WaveletColorsNumberComboBox.ItemIndex := WaveletColorsNumberComboBox.Items.IndexOf(IntToStr(TMPEWaveletGraph(graph).ColorsNumber));
  WaveletShowDataSeriesCheckBox.Checked := TMPEWaveletGraph(graph).ShowDataSeries;
  WaveletInterpolationLevelTrackBarChange(nil);
  end;

// Interface
DataSeriesLineClColorBox.Selected := DATA_SERIES_COLORS[Min(DataSeriesListView.Items.Count, DATA_SERIES_COLORS_COUNT-1)];

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.SaveToProject(Index: Integer = -1);
var
  i: Integer;
  graph: TMPEGraph;
  graphclass: TClass;
  ds: TMPEDataSeries;
begin

// Dodanie/edycja
if Index = -1 then
  begin
  if GraphTypeComboBox.ItemIndex = 1 then
    graphclass := TMPEWaveletGraph
  else
    graphclass := TMPEGraph;
  graph := Project.Graphs.AddGraph(graphclass);
  end
else
  graph := Project.Graphs.Graphs[Index];

// Parametry wykresu
graph.Name := GraphNameEdit.Text;
graph.ShowAxes := GraphShowAxesCheckBox.Checked;
graph.ShowLegend := GraphShowLegendCheckBox.Checked;

graph.AxesColor := GraphAxesClColorBox.Selected;
graph.BackgroundColor := GraphBackgroundClColorBox.Selected;
graph.FontColor := GraphFontClColorBox.Selected;
graph.FontSize := StrToInt(GraphFontSizeComboBox.Text);

graph.AxisXAutomatic := XAxisAutoCheckBox.Checked;
graph.AxisXMin := StrToFloat(XAxisMinEdit.Text);
graph.AxisXMax := StrToFloat(XAxisMaxEdit.Text);

graph.AxisYAutomatic := YAxisAutoCheckBox.Checked;
graph.AxisYMin := StrToFloat(YAxisMinEdit.Text);
graph.AxisYMax := StrToFloat(YAxisMaxEdit.Text);

// Serie danych
graph.DeleteAllDataSeries;

if DataSeriesListView.Items.Count > 0 then
  for i := 0 to DataSeriesListView.Items.Count-1 do
    begin
    ds := TMPEDataSeries(DataSeriesListView.Items[i].Data);
    ds.ImportFromOutputData(Project.OutputData);
    graph.AssignDataSeries(ds);
    end;

// Transformata falkowa
if graph is TMPEWaveletGraph then
  begin
  TMPEWaveletGraph(graph).Wavelet := TMPEWavelet(WaveletWaveletComboBox.ItemIndex);
  TMPEWaveletGraph(graph).SigmaMin := StrToFloat(WaveletSigmaMinEdit.Text);
  TMPEWaveletGraph(graph).SigmaMax := StrToFloat(WaveletSigmaMaxEdit.Text);
  TMPEWaveletGraph(graph).InterpolationLevel := WaveletInterpolationLevelTrackBar.Position / 10;
  TMPEWaveletGraph(graph).ColorTheme := WaveletColorThemeComboBox.ItemIndex;
  TMPEWaveletGraph(graph).ColorsNumber := StrToInt(WaveletColorsNumberComboBox.Items[WaveletColorsNumberComboBox.ItemIndex]);
  TMPEWaveletGraph(graph).ShowDataSeries := WaveletShowDataSeriesCheckBox.Checked;
  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.WaveletInterpolationLevelTrackBarChange(
  Sender: TObject);
begin
WaveletInterpolationLevelLabel.Caption := FloatToStrF(WaveletInterpolationLevelTrackBar.Position / 10, ffFixed, 2, 1) + 'x';
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.XAxisAutoCheckBoxClick(Sender: TObject);
begin
XAxisMinEdit.Enabled := not XAxisAutoCheckBox.Checked;
XAxisMaxEdit.Enabled := not XAxisAutoCheckBox.Checked;
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.YAxisAutoCheckBoxClick(Sender: TObject);
begin
YAxisMinEdit.Enabled := not YAxisAutoCheckBox.Checked;
YAxisMaxEdit.Enabled := not YAxisAutoCheckBox.Checked;
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.AddDataSeriesBtnClick(Sender: TObject);
begin

// Tworzenie serii danych
AddDataSeries(
  DataSeriesNameEdit.Text,
  DataSeriesXAxisComboBox.ItemIndex,
  DataSeriesYAxisComboBox.ItemIndex,
  DataSeriesLineClColorBox.Selected,
  StrToInt(DataSeriesLineWidthComboBox.Text),
  DataSeriesFFT.Checked
  );

// Resetowanie pol edycji
DataSeriesNameEdit.Text := 'new data series';
DataSeriesXAxisComboBox.ItemIndex := 2;
DataSeriesYAxisComboBox.ItemIndex := DataSeriesYAxisComboBox.ItemIndex+1;
DataSeriesAxisComboBoxChange(nil);
DataSeriesLineClColorBox.Selected := DATA_SERIES_COLORS[Min(DataSeriesListView.Items.Count, DATA_SERIES_COLORS_COUNT-1)];
DataSeriesLineWidthComboBox.ItemIndex := DataSeriesLineWidthComboBox.Tag;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DataSeriesAutoNameCheckBoxClick(Sender: TObject);
begin
DataSeriesNameEdit.Enabled := not DataSeriesAutoNameCheckBox.Checked;
DataSeriesAxisComboBoxChange(nil);
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DataSeriesAxisComboBoxChange(Sender: TObject);
begin

if DataSeriesAutoNameCheckBox.Checked then
  begin
  if DataSeriesFFT.Checked then
    DataSeriesNameEdit.Text :=
     'FFT (' + GetColumnFullName(OUTPUT_DATA_COLUMNS, DataSeriesYAxisComboBox.ItemIndex) + ') [Hz]'
  else
    DataSeriesNameEdit.Text :=
      GetColumnFullName(OUTPUT_DATA_COLUMNS, DataSeriesYAxisComboBox.ItemIndex) + ' = f (' +
      GetColumnFullName(OUTPUT_DATA_COLUMNS, DataSeriesXAxisComboBox.ItemIndex) + ')';
  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DataSeriesFFTClick(Sender: TObject);
begin
DataSeriesXAxisComboBox.Enabled := not DataSeriesFFT.Checked;
if DataSeriesFFT.Checked then
  DataSeriesXAxisComboBox.ItemIndex := 2;
DataSeriesAxisComboBoxChange(DataSeriesXAxisComboBox);
end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.DeleteDataSeriesBtnClick(Sender: TObject);
var
  index: Integer;
begin

index := DataSeriesListView.ItemIndex;

if index <> -1 then
  begin
  TMPEDataSeries(DataSeriesListView.Items[index].Data).Free;
  DataSeriesListView.Items.Delete(index);
  DataSeriesListView.ItemIndex := -1;
  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.FormClose(Sender: TObject; var Action: TCloseAction);
var
  i: Integer;
begin

// Zwalnianie nie zatwierdzonych serii danych
if ModalResult <> mrOK then
  if DataSeriesListView.Items.Count > 0 then
    for i := 0 to DataSeriesListView.Items.Count-1 do
      TMPEDataSeries(DataSeriesListView.Items[i].Data).Free;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.FormCreate(Sender: TObject);
var
  b: Byte;
  s: String;
begin

// Aktywna zakladka
PageControl.ActivePageIndex := 0;

// Kolumny pliku wyjsciowego
for b := 0 to OUTPUT_DATA_COLUMNS_COUNT-1 do
  begin
  s := GetColumnFullName(OUTPUT_DATA_COLUMNS, b);
  DataSeriesXAxisComboBox.Items.Add(s);
  DataSeriesYAxisComboBox.Items.Add(s);
  end;

// Interface
GraphTypeComboBoxChange(nil);
WaveletInterpolationLevelTrackBarChange(nil);
DataSeriesXAxisComboBox.ItemIndex := 2;
DataSeriesYAxisComboBox.ItemIndex := 6;
DataSeriesAxisComboBoxChange(nil);

// Schematy kolorow
for b := 0 to GRAPH_THEMES_COUNT-1 do
  GraphThemeComboBox.Items.Add(GRAPH_THEMES[b].Name);
GraphThemeComboBox.ItemIndex := 0;
GraphThemeApplyBtnClick(nil);

// Nazwy falek
for b := 0 to Length(WAVELET_NAMES)-1 do
  WaveletWaveletComboBox.Items.Add(WAVELET_NAMES[TMPEWavelet(b)]);
WaveletWaveletComboBox.ItemIndex := 2;

// Schematy kolorow transformaty falkowej
for b := 0 to WAVELET_THEMES_COUNT-1 do
  WaveletColorThemeComboBox.Items.Add(WAVELET_THEMES[b].Name);
WaveletColorThemeComboBox.ItemIndex := 1;

end;

// -----------------------------------------------------------------------------
procedure TGraphEditForm.GraphThemeApplyBtnClick(Sender: TObject);
var
  i: Integer;
begin

i := GraphThemeComboBox.ItemIndex;

if i <> -1 then
  begin
  GraphBackgroundClColorBox.Selected := GRAPH_THEMES[i].BackgroundColor;
  GraphAxesClColorBox.Selected := GRAPH_THEMES[i].AxesColor;
  GraphFontClColorBox.Selected := GRAPH_THEMES[i].FontColor;
  end;

end;

procedure TGraphEditForm.GraphTypeComboBoxChange(Sender: TObject);
begin

if GraphTypeComboBox.ItemIndex = 1 then
  begin
  WaveletTabSheet.TabVisible := True;
  AxesTabSheet.TabVisible := False;
  GraphBackgroundLabel.Visible := False;
  GraphBackgroundClColorBox.Visible := False;
  end
else
  begin
  WaveletTabSheet.TabVisible := False;
  AxesTabSheet.TabVisible := True;
  GraphBackgroundLabel.Visible := True;
  GraphBackgroundClColorBox.Visible := True;
  end;

end;

end.
