unit MPEGraphsFrame;

interface

// -----------------------------------------------------------------------------
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms, Menus,
  Dialogs, StdCtrls, ComCtrls, ExtCtrls, JPEG, Math,
  MPEExt, MPEGraph, MPEWaveletGraph, MPEGraphEdit, MPEOptions, MPEOutputData,
  MPEProcessMessage, MPEProject;

// -----------------------------------------------------------------------------
type
  TGraphsFrame = class(TFrame)
    GroupBox1: TGroupBox;
    GraphsTabControl: TTabControl;
    AddNewGraphBtn: TButton;
    GroupBox2: TGroupBox;
    ConfigureGraphBtn: TButton;
    GraphPanel: TPanel;
    GraphImage: TImage;
    DeleteGraphBtn: TButton;
    GraphCoordsLabel: TLabel;
    OptionsGraphBtn: TButton;
    GraphExportDataSeriesSaveDialog: TSaveDialog;
    GraphRepaintTimeLabel: TLabel;
    GraphOptionsPopupMenu: TPopupMenu;
    GraphExportDataSeriesMenuItem: TMenuItem;
    GraphExportImageMenuItem: TMenuItem;
    GraphExportImageSaveDialog: TSaveDialog;
    N1: TMenuItem;
    GraphRefreshMenuItem: TMenuItem;
    N2: TMenuItem;
    GraphShowAllMenuItem: TMenuItem;
    GraphReloadMenuItem: TMenuItem;
    OptionsExGraphBtn: TButton;
    GraphOptionsExPopupMenu: TPopupMenu;
    GraphRealoadAllMenuItem: TMenuItem;
    GraphExportImageHighResMenuItem: TMenuItem;
    procedure AddNewGraphBtnClick(Sender: TObject);
    procedure GraphsTabControlChange(Sender: TObject);
    procedure DeleteGraphBtnClick(Sender: TObject);
    procedure ConfigureGraphBtnClick(Sender: TObject);
    procedure GraphImageMouseLeave(Sender: TObject);
    procedure GraphImageMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure GraphImageMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure OptionsGraphBtnClick(Sender: TObject);
    procedure GraphImageMouseEnter(Sender: TObject);
    procedure GraphExportDataSeriesMenuItemClick(Sender: TObject);
    procedure GraphExportImageMenuItemClick(Sender: TObject);
    procedure GraphRefreshMenuItemClick(Sender: TObject);
    procedure GraphImageMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure GraphShowAllMenuItemClick(Sender: TObject);
    procedure GraphReloadMenuItemClick(Sender: TObject);
    procedure OptionsExGraphBtnClick(Sender: TObject);
    procedure GraphRealoadAllMenuItemClick(Sender: TObject);
  private
    // Zmienne
    GraphCoordsLastUpdate: DWord;
    GraphSelectPoint: TPoint;

  public
    // Metody
    procedure ReloadGraph(Index: Integer);
    procedure RepaintGraph(Index: Integer);
    procedure SwitchGraph(Index: Integer);
    procedure EditGraph(Index: Integer);

    procedure LoadGraphsFromProject;

    procedure LoadFromProject;
    procedure SaveToProject;

  end;

// -----------------------------------------------------------------------------
var
  GraphsFrame: TGraphsFrame;

implementation

uses
  MPEMain;

{$R *.dfm}

// -----------------------------------------------------------------------------
procedure TGraphsFrame.ReloadGraph(Index: Integer);
begin
GraphEditForm := TGraphEditForm.Create(Self);
GraphEditForm.LoadFromProject(Index);
GraphEditForm.SaveToProject(Index);
EditGraph(Index);
SwitchGraph(Index);
GraphEditForm.Free;
GraphEditForm := nil;
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.RepaintGraph(Index: Integer);
begin

if (OptionsForm.GraphsQualityComboBox.ItemIndex > 0) or (Project.Graphs.Graphs[Index] is TMPEWaveletGraph) then
  ShowProcessMessage(MainForm);

Project.Graphs.Graphs[Index].Samples := OptionsForm.GraphsQualityComboBox.ItemIndex+1;
Project.Graphs.Graphs[Index].Repaint;

HideProcessMessage;
  
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.SwitchGraph(Index: Integer);
begin
GraphsTabControl.TabIndex := Index;
GraphImage.Picture.Bitmap := Project.Graphs.Graphs[GraphsTabControl.TabIndex].Bitmap;
GraphRepaintTimeLabel.Caption := IntToStr(Project.Graphs.Graphs[GraphsTabControl.TabIndex].RepaintTime) + ' ms';
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.EditGraph(Index: Integer);
begin

if (Index >= 0) and (Index <= GraphsTabControl.Tabs.Count-1) then
  GraphsTabControl.Tabs.Strings[Index] := Project.Graphs.Graphs[Index].Name
else
  GraphsTabControl.Tabs.Add(Project.Graphs.Graphs[Index].Name);

RepaintGraph(Index);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphExportDataSeriesMenuItemClick(Sender: TObject);
begin

if GraphExportDataSeriesSaveDialog.Execute then
  Project.Graphs.Graphs[GraphsTabControl.TabIndex].ExportDataSeriesToFile(GraphExportDataSeriesSaveDialog.FileName);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphExportImageMenuItemClick(Sender: TObject);
var
  graph: TMPEGraph;
  ext: String;
  raw: Boolean;
begin

if GraphExportImageSaveDialog.Execute then
  begin

  graph := Project.Graphs.Graphs[GraphsTabControl.TabIndex];
  ext := LowerCase(ExtractFileExt(GraphExportImageSaveDialog.FileName));
  raw := TMenuItem(Sender).Tag = 1;

  if ext = '.jpg' then
    begin
    with TJPEGImage.Create do
      begin
      if raw then
        Assign(graph.RawBitmap)
      else
        Assign(graph.Bitmap);
      CompressionQuality := 100;
      SaveToFile(GraphExportImageSaveDialog.FileName);
      end;
    end
  else if raw then
    graph.ExportRawImageToFile(GraphExportImageSaveDialog.FileName)
  else
    graph.ExportImageToFile(GraphExportImageSaveDialog.FileName);
    
  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphRealoadAllMenuItemClick(Sender: TObject);
var
  i: Integer;
begin

for i := 0 to GraphsTabControl.Tabs.Count-1 do
  ReloadGraph(i);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphRefreshMenuItemClick(Sender: TObject);
begin

if GraphsTabControl.TabIndex <> -1 then
  begin
  EditGraph(GraphsTabControl.TabIndex);
  SwitchGraph(GraphsTabControl.TabIndex);
  end;
  
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphReloadMenuItemClick(Sender: TObject);
begin

if GraphsTabControl.TabIndex <> -1 then
  ReloadGraph(GraphsTabControl.TabIndex);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphShowAllMenuItemClick(Sender: TObject);
var
  index: Integer;
  graph: TMPEGraph;
begin

index := GraphsTabControl.TabIndex;
graph := Project.Graphs.Graphs[index];

graph.AxisXAutomatic := True;
graph.AxisYAutomatic := True;

EditGraph(index);
SwitchGraph(index);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.OptionsExGraphBtnClick(Sender: TObject);
begin
GraphOptionsExPopupMenu.Popup(Mouse.CursorPos.X, Mouse.CursorPos.Y);
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.OptionsGraphBtnClick(Sender: TObject);
begin
GraphOptionsPopupMenu.Popup(Mouse.CursorPos.X, Mouse.CursorPos.Y);
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.LoadGraphsFromProject;
var
  i: Integer;
begin

GraphsTabControl.Tabs.Clear;

if Project.Graphs.Count > 0 then
  begin

  for i := 0 to Project.Graphs.Count-1 do
    EditGraph(i);

  SwitchGraph(0);
  GraphsTabControl.Visible := True;
  
  end
else
  GraphsTabControl.Visible := False;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.LoadFromProject;
begin

// Dane
LoadGraphsFromProject;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.SaveToProject;
begin

// Dane

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.ConfigureGraphBtnClick(Sender: TObject);
var
  index: Integer;
begin

index := GraphsTabControl.TabIndex;

if index <> -1 then
  begin

  GraphEditForm := TGraphEditForm.Create(Self);
  GraphEditForm.LoadFromProject(index);

  if GraphEditForm.ShowModal = mrOK then
    begin
    GraphEditForm.SaveToProject(index);
    Project.Edited;
    EditGraph(Index);
    SwitchGraph(Index);
    end;

  GraphEditForm.Free;
  GraphEditForm := nil;

  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.DeleteGraphBtnClick(Sender: TObject);
var
  index: Integer;
begin

index := GraphsTabControl.TabIndex;

if index <> -1 then
  if MessageBox(Handle, PChar('Do you really want to remove ' + Project.Graphs.Graphs[index].Name + ' graph?'), PChar(MainForm.Caption), MB_ICONQUESTION or MB_YESNO) = IDYES then
    begin

    Project.Graphs.DeleteGraph(index);
    Project.Edited;
    GraphsTabControl.Tabs.Delete(index);

    if Project.Graphs.Count > 0 then
      SwitchGraph(StepRange(index, 0, Project.Graphs.Count-1))
    else
      GraphsTabControl.Visible := False;

    end;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphImageMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
GraphSelectPoint := Point(X, Y);
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphImageMouseEnter(Sender: TObject);
begin
GraphCoordsLastUpdate := 0;
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphImageMouseLeave(Sender: TObject);
begin
GraphCoordsLabel.Caption := '';
end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphImageMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer);
var
  graph: TMPEGraph;
  prefix: String;
  dx, dy: Double;
begin

if GetTickCount - GraphCoordsLastUpdate >= 20 then
  begin

  graph := Project.Graphs.Graphs[GraphsTabControl.TabIndex];

  if graph is TMPEWaveletGraph then // TODO: Do klasy wykresu?
    prefix := 'Shift x Sigma'
  else
    prefix := 'X x Y';

  graph.PixelsToPoint(X, Y, dx, dy);

  GraphCoordsLabel.Caption :=
    prefix + ' = ' +
    FloatToStrF(dx, ffGeneral, 4, 2) + ' x ' + FloatToStrF(dy, ffGeneral, 4, 2);

  GraphCoordsLastUpdate := GetTickCount;

  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphImageMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  index, f: Integer;
  graph: TMPEGraph;
  dx, dy, sx, sy, delta: Double;
begin

// Wykres
index := GraphsTabControl.TabIndex;
graph := Project.Graphs.Graphs[index];

if not (graph is TMPEWaveletGraph) then
  begin

  // Automatyczne dopasowanie osi
  graph.AxisXAutomatic := False;
  graph.AxisYAutomatic := False;

  // Wspolzedne
  graph.PixelsToPoint(X, Y, dx, dy);

  // Powiekszenie
  if (Button = mbLeft) and ((GraphSelectPoint.X <> X) or (GraphSelectPoint.Y <> Y)) then
    begin

    // Wspolzedne
    graph.PixelsToPoint(GraphSelectPoint.X, GraphSelectPoint.Y, sx, sy);

    // Os X
    graph.AxisXMin := Min(sx, dx);
    graph.AxisXMax := Max(sx, dx);

    // Os Y
    graph.AxisYMin := Min(sy, dy);
    graph.AxisYMax := Max(sy, dy);

    end
  else
    begin

    // Rodzaj powiekszenia
    case Button of
      mbLeft:  f := 4;
      mbRight: f := 1;
      else     f := 2;
    end;

    // Os X
    delta := (graph.AxisXMax-graph.AxisXMin) / f;
    graph.AxisXMin := dx - delta;
    graph.AxisXMax := dx + delta;

    // Os Y
    delta := (graph.AxisYMax-graph.AxisYMin) / f;
    graph.AxisYMin := dy - delta;
    graph.AxisYMax := dy + delta;
    
    end;

  // Odswierzenie
  EditGraph(index);
  SwitchGraph(index);

  end;

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.GraphsTabControlChange(Sender: TObject);
begin

if GraphsTabControl.TabIndex <> -1 then
  SwitchGraph(GraphsTabControl.TabIndex);

end;

// -----------------------------------------------------------------------------
procedure TGraphsFrame.AddNewGraphBtnClick(Sender: TObject);
begin

GraphEditForm := TGraphEditForm.Create(Self);

if GraphsTabControl.Tabs.Count > 0 then
  GraphEditForm.GraphNameEdit.Text := GraphEditForm.GraphNameEdit.Text + ' ' + IntToStr(GraphsTabControl.Tabs.Count+1);

if GraphEditForm.ShowModal = mrOK then
  begin
  GraphEditForm.SaveToProject;
  Project.Edited;
  EditGraph(Project.Graphs.Count-1);
  SwitchGraph(Project.Graphs.Count-1);
  GraphsTabControl.Visible := True;
  end;

GraphEditForm.Free;
GraphEditForm := nil;

end;

end.
