unit MPEOutputData;

interface

// -----------------------------------------------------------------------------
uses
  SysUtils, Classes,
  MPEBuffer, MPEExt;

// -----------------------------------------------------------------------------
type
  TMPEOutputDataRow = class(TObject)
  private
    // Parametry
    FDatas: array[0..OUTPUT_DATA_COLUMNS_COUNT-1] of Double;

    // Metody parametrow
    function GetData(Column: Integer): Double;
    procedure SetData(Column: Integer; Data: Double);

  public
    // Parametry
    property Datas[Column: Integer]: Double read GetData write SetData;

  end;

  TMPEOutputData = class(TObject)
  private
    // Parametry
    FRows: TList;

    // Metody parametrow
    function GetData(Row, Column: Integer): Double;
    procedure SetData(Row, Column: Integer; Data: Double);
    function GetCount: Integer;

  public
    // Parametry
    property Data[Row, Column: Integer]: Double read GetData write SetData;
    property Count: Integer read GetCount;

    // Metody
    function AddRow: TMPEOutputDataRow;
    procedure DeleteAllRows;

    procedure ImportFromFile(const FileName: String);

    procedure LoadFromStream(Stream: TStream);
    procedure SaveToStream(Stream: TStream);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;
    
  end;

implementation

// -----------------------------------------------------------------------------
function TMPEOutputDataRow.GetData(Column: Integer): Double;
begin
Result := FDatas[Column];
end;

// -----------------------------------------------------------------------------
procedure TMPEOutputDataRow.SetData(Column: Integer; Data: Double);
begin
FDatas[Column] := Data;  
end;



// -----------------------------------------------------------------------------
function TMPEOutputData.GetData(Row, Column: Integer): Double;
begin
Result := TMPEOutputDataRow(FRows[Row]).Datas[Column];
end;

// -----------------------------------------------------------------------------
procedure TMPEOutputData.SetData(Row, Column: Integer; Data: Double);
begin
TMPEOutputDataRow(FRows[Row]).Datas[Column] := Data;
end;

// -----------------------------------------------------------------------------
function TMPEOutputData.GetCount: Integer;
begin
Result := FRows.Count;  
end;

// -----------------------------------------------------------------------------
function TMPEOutputData.AddRow: TMPEOutputDataRow;
begin
Result := TMPEOutputDataRow.Create;
FRows.Add(Result);
end;

// -----------------------------------------------------------------------------
procedure TMPEOutputData.DeleteAllRows;
var
  i: Integer;
begin

if FRows.Count > 0 then
  for i := 0 to FRows.Count-1 do
    TMPEOutputDataRow(FRows[i]).Free;

FRows.Clear;

end;

// -----------------------------------------------------------------------------
procedure TMPEOutputData.ImportFromFile(const FileName: String);
var
  b: Byte;
  i: Integer;
  line: String;
  logfile: TStringList;
  params: array[0..OUTPUT_DATA_COLUMNS_COUNT-1] of String;
  row: TMPEOutputDataRow;
begin

// Czyszczenie danych
DeleteAllRows;

// Odczytywanie pliku
logfile := TStringList.Create;
logfile.LoadFromFile(FileName);

// Parsowanie danych
if logfile.Count > 0 then
  for i := 0 to logfile.Count-1 do
    begin

    // Odczytywanie linii
    line := logfile.Strings[i];

    if Length(line) > 0 then
      if line[1] <> OUTPUT_DATA_COMMENT_CHAR then
        begin

        // Oczyszczanie lini
        StripLine(line, OUTPUT_DATA_COMMENT_CHAR);

        // Rozdzielanie wartosci
        Split(' ', line, params);

        // Dodawanie wiersza danych
        row := AddRow;

        for b := 0 to OUTPUT_DATA_COLUMNS_COUNT-1 do
          row.Datas[b] := StrToFloatEx(params[b], OUTPUT_DATA_DECIMAL_SEPARATOR);

        end;

    end;

// Zwalnianie listy danych pliku
logfile.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEOutputData.LoadFromStream(Stream: TStream);
var
  buffer: TMPEBuffer;
  i, u, rcount: Integer;
  datarow: TMPEOutputDataRow;
begin

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Czyszczenie danych
DeleteAllRows;

// Dane wierszy danych
rcount := buffer.ReadInteger;

if rcount > 0 then
  for i := 0 to rcount-1 do
    begin
    datarow := AddRow;
    for u := 0 to OUTPUT_DATA_COLUMNS_COUNT-1 do
      datarow.Datas[u] := buffer.ReadDouble;
    end;

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEOutputData.SaveToStream(Stream: TStream);
var
  buffer: TMPEBuffer;
  i, u: Integer;
begin

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Dane wierszy danych
buffer.WriteData(Count);

if Count > 0 then
  for i := 0 to Count-1 do
    for u := 0 to OUTPUT_DATA_COLUMNS_COUNT-1 do
      buffer.WriteData(Data[i, u]);

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEOutputData.Create;
begin

inherited Create;

// Parametry
FRows := TList.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEOutputData.Destroy;
begin

// Czyszczenie wierszy
DeleteAllRows;

// Parametry
FRows.Free;

inherited Destroy;

end;

end.
