unit MPEProject;

interface

// -----------------------------------------------------------------------------
uses
  SysUtils, Classes,
  MPEBuffer, MPEExt, MPEGraph, MPEWaveletGraph, MPEGraphs, MPEOutputData, MPEAnimations;

// -----------------------------------------------------------------------------
type
  TMPEProject = class(TObject)
  private
    // Parametry
    FName: String;
    FComment: String;
    FPath: String;

    FSaved: Boolean;

    FGraphs: TMPEGraphs;
    FOutputData: TMPEOutputData;
    FAnimations: TMPEAnimations;

    // Metody parametrow
    procedure SetPath(const Path: String);
    function GetSimName: String;

  public
    // Parametry
    property Name: String read FName write FName;
    property Comment: String read FComment write FComment;
    property Path: String read FPath write SetPath;

    property SimName: String read GetSimName;
    property Saved: Boolean Read FSaved;

    property Graphs: TMPEGraphs read FGraphs;
    property OutputData: TMPEOutputData read FOutputData;
    property Animations: TMPEAnimations read FAnimations;

    // Metody
    procedure Edited;
    procedure Loaded;

    function LoadFromFile(const FileName: String): Boolean;
    procedure SaveToFile(const FileName: String);

    // Tworzenie
    constructor Create;
    destructor Destroy; override;
  
  end;

// -----------------------------------------------------------------------------
var
  Project: TMPEProject;

implementation

// -----------------------------------------------------------------------------
procedure TMPEProject.SetPath(const Path: String);
begin
if Trim(Path) <> '' then
  FPath := StripSlash(Path) + '\'
else
  FPath := '';
end;

// -----------------------------------------------------------------------------
function TMPEProject.GetSimName: String;
var
  i: Integer;
  line: String;
  param, value: String;
  allopt: TStringList;
begin

if (FPath <> '') and FileExists(FPath + INPUT_DATA_FILENAME) then
  begin

  // Odczytywanie pliku
  allopt := TStringList.Create;
  allopt.LoadFromFile(FPath + INPUT_DATA_FILENAME);

  // Parsowanie danych
  if allopt.Count > 0 then
    for i := 0 to allopt.Count-1 do
      begin

      // Odczytywanie linii
      line := allopt.Strings[i];

      if Length(line) > 0 then
        if line[1] = INPUT_DATA_PARAM_CHAR then
          begin

          // Oczyszczanie lini
          StripLine(line, INPUT_DATA_COMMENT_CHAR);

          // Parametr, wartosc
          SplitParamValue(line, param, value);

          // Nazwa symulacji
          if param = INPUT_DATA_SIMNAME_PARAM_NAME then
            begin
            Result := value;
            Break;
            end;

          end;

      end;

  // Zwalnianie listy danych pliku
  allopt.Free;

  end
else
  Result := '';

end;

// -----------------------------------------------------------------------------
procedure TMPEProject.Edited;
begin
FSaved := False;
end;

// -----------------------------------------------------------------------------
procedure TMPEProject.Loaded;
begin
FSaved := True;
end;

// -----------------------------------------------------------------------------
function TMPEProject.LoadFromFile(const FileName: String): Boolean;
var
  stream, gstream, odstream: TMemoryStream;
  buffer, gbuffer: TMPEBuffer;
  i, gcount: Integer;
  version: Byte;
  graph: TMPEGraph;
begin

// Tworzenie i odczyt strumienia
stream := TMemoryStream.Create;
stream.LoadFromFile(FileName);

// Tworzenie bufora
buffer := TMPEBuffer.Create(stream);

// Wersja
version := buffer.ReadByte;

if version = FILE_VERSION then
  begin

  // Dane projektu
  FName := buffer.ReadString;
  FComment := buffer.ReadString;
  FPath := buffer.ReadString;

  // Dane wykresow
  gcount := buffer.ReadInteger;

  gstream := TMemoryStream.Create;
  gbuffer := TMPEBuffer.Create(gstream);

  buffer.ReadStream(gstream);

  if gcount > 0 then
    for i := 0 to gcount-1 do
      begin
      if gbuffer.ReadString = 'TMPEWaveletGraph' then
        graph := FGraphs.AddGraph(TMPEWaveletGraph)
      else
        graph := FGraphs.AddGraph(TMPEGraph);
      graph.LoadFromStream(gstream);
      end;

  gbuffer.Free;
  gstream.Free;

  // Dane danych wyjsciowych
  odstream := TMemoryStream.Create;
  buffer.ReadStream(odstream);
  FOutputData.LoadFromStream(odstream);
  odstream.Free;

  // Wynik
  Result := True;

  end
else
  Result := False;

// Zwalnianie bufora
buffer.Free;

// Zwolnienie strumienia
stream.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEProject.SaveToFile(const FileName: String);
var
  stream, gstream, odstream: TMemoryStream;
  buffer, gbuffer: TMPEBuffer;
  i: Integer;
begin

// Tworzenie strumienia
stream := TMemoryStream.Create;

// Tworzenie bufora
buffer := TMPEBuffer.Create(stream);

// Wersja
buffer.WriteData(FILE_VERSION);

// Dane projektu
buffer.WriteData(FName);
buffer.WriteData(FComment);
buffer.WriteData(FPath);

// Dane wykresow
buffer.WriteData(FGraphs.Count);

gstream := TMemoryStream.Create;
gbuffer := TMPEBuffer.Create(gstream);

if FGraphs.Count > 0 then
  for i := 0 to FGraphs.Count-1 do
    begin
    gbuffer.WriteData(FGraphs.Graphs[i].ClassName);
    FGraphs.Graphs[i].SaveToStream(gstream);
    end;

buffer.WriteData(gstream, cmLZMA);

gbuffer.Free;
gstream.Free;

// Dane danych wyjsciowych
odstream := TMemoryStream.Create;

FOutputData.SaveToStream(odstream);

buffer.WriteData(odstream, cmLZMA);
odstream.Free;

// Zwalnianie bufora
buffer.Free;

// Zapis i zwolnienie strumienia
stream.SaveToFile(FileName);
stream.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEProject.Create;
begin

inherited Create;

// Parametry
FName := 'New project';
FComment := '';
FPath := '';

FSaved := True;

FGraphs := TMPEGraphs.Create;
FOutputData := TMPEOutputData.Create;
FAnimations := TMPEAnimations.Create;

end;

// -----------------------------------------------------------------------------
destructor TMPEProject.Destroy;
begin

// Parametry
FGraphs.Free;
FOutputData.Free;
FAnimations.Free;

inherited Destroy;

end;

end.
