unit MPEWaveletGraph;

interface

// -----------------------------------------------------------------------------
uses
  Windows, Classes, SysUtils, Graphics,
  Math, Resample,
  MPEBuffer, MPEExt, MPEGraph;

// -----------------------------------------------------------------------------
type
  TMPEWavelet = (wlNone, wlRectangle, wlGauss, wlGaussFirstDerivative, wlGaussSecondDerivative, wlMorlet, wlMexicanHat);

  TMPEWaveletFunc = function(Time, Sigma: Single): Single of object;

  TMPEWaveletSigma = record
    Value: Single;
    Sqr: Single;
    Sqrt: Single;
  end;

  TWaveletTheme = record
    Name: String;
    BorderColor: Integer;
    FontColor: Integer;
    Colors: array[0..255] of Integer;
  end;

// -----------------------------------------------------------------------------
const
  WAVELET_NAMES: array[TMPEWavelet] of String = ('None', 'Rectangle', 'Gauss', 'Gauss first derivative', 'Gauss second derivative', 'Morlet', 'Mexican Hat');

  WAVELET_THEMES_COUNT = 3;
  WAVELET_THEMES: array[0..WAVELET_THEMES_COUNT-1] of TWaveletTheme = (
    (Name: 'Grayscale'; BorderColor: $808080; FontColor: $808080; Colors: (
      $000000, $010101, $020202, $030303, $040404, $050505, $060606, $070707,
      $080808, $090909, $0A0A0A, $0B0B0B, $0C0C0C, $0D0D0D, $0E0E0E, $0F0F0F,
      $101010, $111111, $121212, $131313, $141414, $151515, $161616, $171717,
      $181818, $191919, $1A1A1A, $1B1B1B, $1C1C1C, $1D1D1D, $1E1E1E, $1F1F1F,
      $202020, $212121, $222222, $232323, $242424, $252525, $262626, $272727,
      $282828, $292929, $2A2A2A, $2B2B2B, $2C2C2C, $2D2D2D, $2E2E2E, $2F2F2F,
      $303030, $313131, $323232, $333333, $343434, $353535, $363636, $373737,
      $383838, $393939, $3A3A3A, $3B3B3B, $3C3C3C, $3D3D3D, $3E3E3E, $3F3F3F,
      $404040, $414141, $424242, $434343, $444444, $454545, $464646, $474747,
      $484848, $494949, $4A4A4A, $4B4B4B, $4C4C4C, $4D4D4D, $4E4E4E, $4F4F4F,
      $505050, $515151, $525252, $535353, $545454, $555555, $565656, $575757,
      $585858, $595959, $5A5A5A, $5B5B5B, $5C5C5C, $5D5D5D, $5E5E5E, $5F5F5F,
      $606060, $616161, $626262, $636363, $646464, $656565, $666666, $676767,
      $686868, $696969, $6A6A6A, $6B6B6B, $6C6C6C, $6D6D6D, $6E6E6E, $6F6F6F,
      $707070, $717171, $727272, $737373, $747474, $757575, $767676, $777777,
      $787878, $797979, $7A7A7A, $7B7B7B, $7C7C7C, $7D7D7D, $7E7E7E, $7F7F7F,
      $808080, $818181, $828282, $838383, $848484, $858585, $868686, $878787,
      $888888, $898989, $8A8A8A, $8B8B8B, $8C8C8C, $8D8D8D, $8E8E8E, $8F8F8F,
      $909090, $919191, $929292, $939393, $949494, $959595, $969696, $979797,
      $989898, $999999, $9A9A9A, $9B9B9B, $9C9C9C, $9D9D9D, $9E9E9E, $9F9F9F,
      $A0A0A0, $A1A1A1, $A2A2A2, $A3A3A3, $A4A4A4, $A5A5A5, $A6A6A6, $A7A7A7,
      $A8A8A8, $A9A9A9, $AAAAAA, $ABABAB, $ACACAC, $ADADAD, $AEAEAE, $AFAFAF,
      $B0B0B0, $B1B1B1, $B2B2B2, $B3B3B3, $B4B4B4, $B5B5B5, $B6B6B6, $B7B7B7,
      $B8B8B8, $B9B9B9, $BABABA, $BBBBBB, $BCBCBC, $BDBDBD, $BEBEBE, $BFBFBF,
      $C0C0C0, $C1C1C1, $C2C2C2, $C3C3C3, $C4C4C4, $C5C5C5, $C6C6C6, $C7C7C7,
      $C8C8C8, $C9C9C9, $CACACA, $CBCBCB, $CCCCCC, $CDCDCD, $CECECE, $CFCFCF,
      $D0D0D0, $D1D1D1, $D2D2D2, $D3D3D3, $D4D4D4, $D5D5D5, $D6D6D6, $D7D7D7,
      $D8D8D8, $D9D9D9, $DADADA, $DBDBDB, $DCDCDC, $DDDDDD, $DEDEDE, $DFDFDF,
      $E0E0E0, $E1E1E1, $E2E2E2, $E3E3E3, $E4E4E4, $E5E5E5, $E6E6E6, $E7E7E7,
      $E8E8E8, $E9E9E9, $EAEAEA, $EBEBEB, $ECECEC, $EDEDED, $EEEEEE, $EFEFEF,
      $F0F0F0, $F1F1F1, $F2F2F2, $F3F3F3, $F4F4F4, $F5F5F5, $F6F6F6, $F7F7F7,
      $F8F8F8, $F9F9F9, $FAFAFA, $FBFBFB, $FCFCFC, $FDFDFD, $FEFEFE, $FFFFFF
      )
    ),
    (Name: 'Rainbow'; BorderColor: $303030; FontColor: $303030; Colors: (
      $CC29A3, $CD2B9F, $CE2E9B, $CF3097, $D03393, $D13690, $D3388C, $D43B88,
      $D53E84, $D64081, $D7437D, $D84679, $DA4875, $DB4B72, $DC4E6E, $DD506A,
      $DE5366, $DF5663, $E1585F, $E25B5B, $E35E57, $E46054, $E56350, $E7664C,
      $E86848, $E96B44, $EA6D41, $EB703D, $EC7339, $EE7535, $EF7832, $F07B2E,
      $F17D2A, $F28026, $F38323, $F5851F, $F6881B, $F78B17, $F88D14, $F99010,
      $FB930C, $FC9508, $FD9805, $FE9B01, $FE9D00, $FEA000, $FEA200, $FEA400,
      $FEA700, $FEA900, $FEAC00, $FEAE00, $FEB000, $FEB300, $FEB500, $FEB800,
      $FEBA00, $FEBD00, $FEBF00, $FEC100, $FEC400, $FEC600, $FEC900, $FECB00,
      $FECE00, $FED000, $FED200, $FED500, $FED700, $FEDA00, $FEDC00, $FEDF00,
      $FEE100, $FEE300, $FEE600, $FEE800, $FEEB00, $FEED00, $FEF000, $FEF200,
      $FEF400, $FEF700, $FEF900, $FEFC00, $FEFE00, $F9FE00, $F4FE00, $EEFE00,
      $E8FE00, $E2FE00, $DCFE00, $D6FE00, $D0FF00, $CAFE00, $C5FE00, $BFFE00,
      $B9FE00, $B3FE00, $ADFE00, $A7FE00, $A1FE00, $9BFE00, $95FE00, $90FE00,
      $8AFE00, $84FE00, $7EFE00, $78FE00, $72FE00, $6CFE00, $66FE00, $61FE00,
      $5BFE00, $55FE00, $4FFE00, $49FE00, $43FE00, $3DFE00, $37FE00, $31FE00,
      $2CFE00, $26FE00, $20FE00, $1AFE00, $14FE00, $0EFE00, $08FE00, $02FE00,
      $00FE02, $00FE08, $00FE0D, $00FE13, $00FE19, $00FE1E, $00FE24, $00FE29,
      $00FE2F, $00FE34, $00FE3A, $00FE3F, $00FE45, $00FE4B, $00FE50, $00FE56,
      $00FE5B, $00FE61, $00FE66, $00FE6C, $00FE71, $00FE77, $00FE7D, $00FE82,
      $00FE88, $00FE8D, $00FE93, $00FE98, $00FE9E, $00FEA3, $00FEA9, $00FEAF,
      $00FEB4, $00FEBA, $00FEBF, $00FEC5, $00FECA, $00FED0, $00FED5, $00FEDB,
      $00FEE0, $00FEE6, $00FEEC, $00FEF1, $00FEF7, $00FEFC, $00FCFE, $00F8FE,
      $00F5FE, $00F1FE, $00EDFE, $00E9FE, $00E6FE, $00E2FE, $00DEFE, $00DAFE,
      $00D7FE, $00D3FE, $00CFFE, $00CBFE, $00C8FE, $00C4FE, $00C0FE, $00BCFE,
      $00B9FE, $00B5FE, $00B1FE, $00ADFE, $00AAFE, $00A6FE, $00A2FE, $009EFE,
      $009BFE, $0097FE, $0093FE, $008FFE, $008CFE, $0088FE, $0084FE, $0080FE,
      $007DFE, $0079FE, $0075FE, $0071FE, $006EFE, $006AFE, $0066FE, $0064FF,
      $0061FF, $005FFF, $005CFF, $005AFF, $0057FF, $0055FF, $0052FF, $0050FF,
      $004DFF, $004BFF, $0048FF, $0046FF, $0043FF, $0041FF, $003EFF, $003CFF,
      $0039FF, $0037FF, $0034FF, $0032FF, $002FFF, $002DFF, $002AFF, $0028FF,
      $0025FF, $0023FF, $0020FF, $001EFF, $001BFF, $0018FE, $0016FE, $0013FE,
      $0011FE, $000EFE, $000CFE, $0009FE, $0007FE, $0005FF, $0002FF, $0000FF
      )
    ),
    (Name: 'Heat'; BorderColor: $303030; FontColor: $303030; Colors: (
      $FFFFFF, $FDFEFF, $FBFDFF, $F8FCFE, $F6FBFE, $F4FAFE, $F2F9FE, $F0F8FE,
      $EEF7FE, $ECF6FE, $EAF5FE, $E9F4FF, $E7F3FF, $E5F2FF, $E3F1FF, $E1F0FF, 
      $DFEFFF, $DDEEFF, $DBEDFF, $D9ECFF, $D7EBFF, $D5EAFF, $D3E9FF, $D1E8FF, 
      $CFE7FF, $CDE6FF, $CBE5FF, $C9E4FF, $C7E3FF, $C5E2FF, $C3E1FF, $C1E0FF, 
      $BFDFFF, $BDDEFF, $BBDDFF, $B9DCFF, $B7DBFF, $B5DAFF, $B3D9FF, $B1D8FF, 
      $AFD7FF, $ACD6FE, $AAD5FE, $A8D4FE, $A6D3FE, $A4D2FE, $A2D1FE, $A0D0FE, 
      $9ECFFE, $9CCEFE, $9ACDFE, $98CCFE, $96CBFE, $94CAFE, $92C9FE, $90C8FE, 
      $8EC7FE, $8CC6FE, $8AC5FE, $88C4FE, $86C3FE, $84C2FE, $82C1FE, $80C0FE, 
      $7EBFFE, $7CBEFE, $7ABDFE, $78BCFE, $76BBFE, $74BAFE, $72B9FE, $70B8FE, 
      $6EB7FE, $6CB6FE, $6AB5FE, $68B4FE, $66B3FE, $64B2FE, $62B1FE, $60B0FE, 
      $5EAFFE, $5CAEFE, $5BADFF, $58ACFE, $56ABFE, $54AAFE, $52A9FE, $50A8FE, 
      $4EA7FE, $4CA6FE, $4AA5FE, $49A4FE, $47A3FE, $45A2FE, $43A1FE, $41A0FE, 
      $3F9FFE, $3D9EFE, $3B9DFE, $399CFE, $379BFE, $359AFE, $3399FE, $3198FE, 
      $2F97FE, $2D96FE, $2B95FE, $2994FE, $2793FE, $2592FE, $2391FE, $2190FE, 
      $1F8FFE, $1D8EFE, $1B8DFE, $198CFE, $178BFE, $158AFE, $1389FE, $1188FE, 
      $0F87FE, $0D86FE, $0B85FE, $0984FE, $0783FE, $0582FE, $0381FE, $0180FE, 
      $007FFE, $007EFE, $007DFD, $007CFD, $007BFD, $007AFC, $0079FC, $0078FB, 
      $0077FB, $0076FB, $0075FA, $0074FA, $0073F9, $0072F9, $0071F9, $0070F8, 
      $006FF8, $006EF7, $006DF7, $006CF7, $006BF6, $006AF6, $0069F5, $0068F5, 
      $0067F5, $0066F4, $0065F4, $0064F3, $0063F3, $0062F3, $0061F2, $0060F2, 
      $005FF1, $005EF1, $005DF1, $005CF0, $005BF0, $005AEF, $0059EF, $0058EF, 
      $0057EE, $0056EE, $0055ED, $0054ED, $0053ED, $0052EC, $0051EC, $0050EB, 
      $004FEB, $004EEB, $004DEA, $004CEA, $004BE9, $004AE9, $0049E9, $0048E8, 
      $0047E8, $0046E7, $0045E7, $0044E7, $0043E6, $0042E6, $0041E5, $0040E5, 
      $003FE5, $003EE4, $003DE4, $003CE3, $003BE3, $003AE3, $0039E2, $0038E2, 
      $0037E1, $0036E1, $0035E1, $0034E0, $0033E0, $0032DF, $0031DF, $0030DF, 
      $002FDE, $002EDE, $002DDD, $002CDD, $002BDD, $002ADC, $0029DC, $0028DB, 
      $0027DB, $0026DB, $0025DA, $0024DA, $0023D9, $0022D9, $0021D9, $0020D8, 
      $001FD8, $001ED7, $001DD7, $001CD7, $001BD6, $001AD6, $0019D5, $0018D5, 
      $0017D5, $0016D4, $0015D4, $0014D3, $0013D3, $0012D3, $0011D2, $0010D2, 
      $000FD1, $000ED1, $000DD1, $000CD0, $000BD0, $000ACF, $0009CF, $0008CF, 
      $0007CE, $0006CE, $0005CD, $0004CD, $0003CD, $0002CC, $0001CC, $0000CC
      )
    )
    );

// -----------------------------------------------------------------------------
type
  TMPEWaveletGraph = class(TMPEGraph)
  const
    // Stale
    THREE_0_5: Single = 1.7320508075688772935274463415059; // 3^0.5
    PI_0_25:   Single = 1.3313353638003897127975349179503; // PI^0.25
    PI_0_5:    Single = 1.7724538509055160272981674833411; // PI^0.5
    PI_2:      Single = 9.8696044010893586188344909998762; // PI^2
    PI_2_0_5:  Single = 2.5066282746310005024157652848110; // (PI*2)^0.5

  private
    // Zmienne
    Sigma: TMPEWaveletSigma;

    // Metody
    function WaveletNone(Time, Sigma: Single): Single;
    function WaveletRectangle(Time, Sigma: Single): Single;
    function WaveletGauss(Time, Sigma: Single): Single;
    function WaveletGaussFirstDerivative(Time, Sigma: Single): Single;
    function WaveletGaussSecondDerivative(Time, Sigma: Single): Single;
    function WaveletMorlet(Time, Sigma: Single): Single;
    function WaveletMexicanHat(Time, Sigma: Single): Single;

  protected
    // Parametry
    FWavelet: TMPEWavelet;
    FSigmaMin: Single;
    FSigmaMax: Single;
    FInterpolationLevel: Single;
    FColorTheme: Integer;
    FColorsNumber: Word;
    FShowDataSeries: Boolean;

    // Metody parametrow
    procedure SetSigmaMin(SigmaMin: Single);
    procedure SetSigmaMax(SigmaMax: Single);
    procedure SetInterpolationLevel(InterpolationLevel: Single);

    // Metody
    procedure FitAxes(XMargin: Single = 0; YMargin: Single = 0); override;
    procedure DrawBackground(Buffer: TBitmap); override;
    procedure DrawAxes(Buffer: TBitmap); override;
    procedure DrawDataSeries(Buffer: TBitmap); override;
    procedure DrawLegend(Buffer: TBitmap; Top: Integer = 0); override;

  public
    // Parametry
    property Wavelet: TMPEWavelet read FWavelet write FWavelet;
    property SigmaMin: Single read FSigmaMin write SetSigmaMin;
    property SigmaMax: Single read FSigmaMax write SetSigmaMax;
    property InterpolationLevel: Single read FInterpolationLevel write SetInterpolationLevel;
    property ColorTheme: Integer read FColorTheme write FColorTheme;
    property ColorsNumber: Word read FColorsNumber write FColorsNumber;
    property ShowDataSeries: Boolean read FShowDataSeries write FShowDataSeries;

    // Metody
    procedure PixelsToPoint(Left, Top: Integer; var X, Y: Double); override;

    procedure LoadFromStream(Stream: TStream); override;
    procedure SaveToStream(Stream: TStream); override;

    // Tworzenie
    constructor Create; override;
    
  end;

implementation

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletNone(Time, Sigma: Single): Single;
begin
Result := 1;
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletRectangle(Time, Sigma: Single): Single;
begin
if (Time >= -Sigma) and (Time <= Sigma) then
  Result := 1 / (2*Sigma)
else
  Result := 0;
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletGauss(Time, Sigma: Single): Single;
begin
Result := Exp(-Sqr(Time)/(2*Self.Sigma.Sqr)) / (PI_2_0_5*Sigma);
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletGaussFirstDerivative(Time, Sigma: Single): Single;
begin
Result := WaveletGauss(Time, Sigma) * (-Time/Self.Sigma.Sqr);
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletGaussSecondDerivative(Time, Sigma: Single): Single;
begin
Result := WaveletGauss(Time, Sigma) * ((Sqr(Time)-Self.Sigma.Sqr)/Sqr(Self.Sigma.Sqr));
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletMorlet(Time, Sigma: Single): Single;
var
  time_pi_sigma_2: Single;
begin
time_pi_sigma_2 := -2*Sqr(Time)*PI_2/Self.Sigma.Sqr;
Result := Cos(2*PI*Time) * Exp(time_pi_sigma_2) - Exp(-0.5*Self.Sigma.Sqr+time_pi_sigma_2);
end;

// -----------------------------------------------------------------------------
function TMPEWaveletGraph.WaveletMexicanHat(Time, Sigma: Single): Single;
var
  time_sigma_2: Single;
begin
time_sigma_2 := Sqr(Time/Sigma);  
Result := (2 / (Self.Sigma.Sqrt*THREE_0_5*PI_0_25)) * (1-time_sigma_2) * Exp(-0.5*time_sigma_2);
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.SetSigmaMin(SigmaMin: Single);
begin
if SigmaMin <= 0 then
  FSigmaMin := 1
else
  FSigmaMin := SigmaMin;
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.SetSigmaMax(SigmaMax: Single);
begin
if SigmaMax <= 0 then
  FSigmaMax := 1
else
  FSigmaMax := SigmaMax;
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.SetInterpolationLevel(InterpolationLevel: Single);
begin
FInterpolationLevel := Max(Min(InterpolationLevel, 10), 1);
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.FitAxes(XMargin: Single = 0; YMargin: Single = 0);
begin
inherited FitAxes(0, 0.02);
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.DrawBackground(Buffer: TBitmap);
type
  PSingleArray = ^TSingleArray;
  TSingleArray = array[0..32767] of Single;
var
  i, x, y, width, height, index, count, zero_count, color: Integer;
  ratiox, ratioy, sx, sy, a, b, c: Single;
  int, maxint, minint: Single;
  xval, lastxval, xmin, xmax, xshiftfactor, xshift, time, maxtime: Single;
  wval, lastwval: Single;
  sigmafactor: Single;
  line: PByteArray;
  datas, values: PSingleArray;
  series: TMPEDataSeries;
  wavelet: TMPEWaveletFunc;
begin

// Czyszczenie bufora wyjsciowego
Buffer.Canvas.Brush.Color := WAVELET_THEMES[FColorTheme].Colors[0];
Buffer.Canvas.FillRect(Bounds(0, 0, Buffer.Width, Buffer.Height));

// Serie danych
if (DataSeriesCount > 0) and (DataSeries[0].DataPointsCount >= 2) then
  begin

  // Inicjalizacja
  width := Round(Buffer.Width / (FInterpolationLevel*FSamples));
  height := Round(Buffer.Height / (FInterpolationLevel*FSamples));
  ratiox := (width-1) / Buffer.Width;
  ratioy := (height-1) / Buffer.Height;

  // Falka
  case FWavelet of
    wlNone: wavelet := WaveletNone;
    wlRectangle: wavelet := WaveletRectangle;
    wlGauss: wavelet := WaveletGauss;
    wlGaussFirstDerivative: wavelet := WaveletGaussFirstDerivative;
    wlGaussSecondDerivative: wavelet := WaveletGaussSecondDerivative;
    wlMorlet: wavelet := WaveletMorlet;
    wlMexicanHat: wavelet := WaveletMexicanHat;
    else wavelet := WaveletNone;
  end;
         
  // Bufor danych
  series := DataSeries[0];
  count := series.DataPointsCount;
  datas := AllocMem(count*2*SizeOf(Single));
  for i := 0 to count-1 do
    begin
    datas^[i*2+0] := series.DataPoints[i].X;
    datas^[i*2+1] := series.DataPoints[i].Y;
    end;

  // Bufor wynikow
  values := AllocMem(width*height*SizeOf(Single));

  // Wartosci minimalne, maksymalne
  xmin := datas^[0];
  xmax := datas^[(count-1)*2];
  minint := 0;
  maxint := 0;
  maxtime := 0;

  // Wspolczynniki
  xshiftfactor := (xmax - xmin) / (width-1);
  sigmafactor := (FSigmaMax-FSigmaMin) / (height-1);

  // Transformata falkowa
  for y := 0 to height-1 do
    begin

    // Skala
    Sigma.Value := y * sigmafactor + FSigmaMin;
    Sigma.Sqr := Sqr(Sigma.Value);
    Sigma.Sqrt := Sqrt(Sigma.Value);

    // Badanie zakresu zmiennosci falki
    if maxtime < xmax-xmin then
      begin
      maxtime := xmax-xmin;
      zero_count := 0;
      for x := 0 to width-1 do
        begin
        time := x*xshiftfactor;
        wval := wavelet(time, Sigma.Value);
        if wval <> 0 then
          zero_count := 0
        else if zero_count >= 4 then
          break
        else
          begin
          maxtime := time;
          Inc(zero_count);
          end;
        end;
      end;

    // Zmiana przesuniecia
    for x := 0 to width-1 do
      begin

      // Przesuniecie
      xshift := x * xshiftfactor;

      // Calkowanie
      int := 0;
      lastxval := xmin;
      lastwval := 0;
      
      for i := 0 to count-1 do
        begin
        xval := datas^[i*2];
        if (xval = lastxval) and (i > 0) then
          continue;
        time := xval-xmin-xshift;
        if time < -maxtime then
          continue
        else if time > maxtime then
          break;
        wval := wavelet(time, Sigma.Value)*datas^[i*2+1];
        if i > 0 then
          int := int + ((lastwval+wval) * (xval-lastxval) * 0.5);
        lastxval := xval;
        lastwval := wval;
        end;

      int := int / Sigma.Sqrt;

      // Wartosc minimalna i maksymalna
      minint := Min(minint, int);
      maxint := Max(maxint, int);

      // Zapisanie wyniku
      values^[y*width + x] := int;

      end;

    end;

  // Normalizacja
  if minint = maxint then
    begin
    minint := 0;
    if maxint = 0 then
      maxint := 1;
    end;
  for i := 0 to width*height-1 do
    values^[i] := (values^[i]-minint) / (maxint-minint);

  // Rzutowanie bufora wynikow
  {for x := 0 to width-1 do
  for y := 0 to height-1 do
    begin
    x1 := Floor(x/ratiox);
    x2 := Min(Floor((x+1)/ratiox), Buffer.Width-1);
    y1 := Floor(y/ratioy);
    y2 := Min(Floor((y+1)/ratioy), Buffer.Height-1);
    index := (height-1-y)*width + x;
    for u := y1 to y2 do
      begin
      line := Buffer.ScanLine[u];
      if y1 = y2 then
        begin
        a := values^[index];
        b := values^[index+1];
        end
      else
        begin
        a := ((u-y1)*(values^[index-width]  -values^[index])   / (y2-y1)) + values^[index];
        b := ((u-y1)*(values^[index-width+1]-values^[index+1]) / (y2-y1)) + values^[index+1];
        end;
      for i := x1 to x2 do
        begin
        if x1 = x2 then
          c := a
        else
          c := ((i-x1)*(b-a) / (x2-x1)) + a;
        color := Min(Round(c*16)*16, 255);
        line^[i*3+0] := color;
        line^[i*3+1] := color;
        line^[i*3+2] := color;
        end;
      end;
    end;}

  // Rzutowanie bufora wynikow
  for y := 0 to Buffer.Height-1 do
    begin
    line := Buffer.ScanLine[y];
    for x := 0 to Buffer.Width-1 do
      begin
      if (FInterpolationLevel > 1) or (FSamples > 1) then
        begin
        sx := ratiox*x;
        sy := ratioy*y;
        a := sx - Trunc(sx);
        b := sy - Trunc(sy);
        index := (height-1-Trunc(sy))*width + Trunc(sx);
        c :=
          (1-a)*(1-b)*values^[index] +
          (  a)*(1-b)*values^[index+1] +
          (1-a)*(  b)*values^[index-width] +
          (  a)*(  b)*values^[index-width+1];
        end
      else
        c := values^[(height-1-y)*width + x];
      color := WAVELET_THEMES[FColorTheme].Colors[Floor(Round(c*(FColorsNumber-1))*(255/(FColorsNumber-1)))];
      line^[x*3+0] := color shr 16;
      line^[x*3+1] := (color shr 8) and $FF;
      line^[x*3+2] := color and $FF;
      end;
    end;

  // Zwalnianie buforow
  FreeMem(datas);
  FreeMem(values);

  end;

end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.DrawAxes(Buffer: TBitmap);
var
  i: Integer;
  f: Double;
  s: String;
  point, axes: TPoint;
begin

if FShowDataSeries then
  begin

  // Osie
  axes := Classes.Point(0, FHeight*FSamples-1);

  // Parametry osi
  Buffer.Canvas.Brush.Style := bsClear;

  Buffer.Canvas.Pen.Color := FAxesColor;
  Buffer.Canvas.Pen.Width := Ceil(0.5*FSamples);

  Buffer.Canvas.Font.Color := FFontColor;
  Buffer.Canvas.Font.Size := Ceil(FFontSize*FSamples);

  // Os odcietych
  Buffer.Canvas.MoveTo(0, axes.Y);
  Buffer.Canvas.LineTo(FWidth*FSamples, axes.Y);

  // Os rzednych
  Buffer.Canvas.MoveTo(axes.X, 0);
  Buffer.Canvas.LineTo(axes.X, FHeight*FSamples);

  // Wartosci osi odcietych
  f := (FAxisXMax-FAxisXMin) / (FWidth/60);
  for i := Round(FAxisXMin / f) to Round(FAxisXMax / f) do
    begin
            
    point := ValueToPixels(i*f, 0, FAxisXMin, FAxisXMax, FAxisYMin, FAxisYMax);

    if (point.X >= 0) and (point.X <= FWidth*FSamples-1) then
      begin
      Buffer.Canvas.MoveTo(point.X, axes.Y);
      Buffer.Canvas.LineTo(point.X, axes.Y-8*FSamples);
      s := FloatToStrF(i*f, ffGeneral, 3, 2);
      Buffer.Canvas.TextOut(point.X-(Buffer.Canvas.TextWidth(s) div 2), axes.Y-(Buffer.Canvas.TextHeight(s)+10*FSamples), s);
      end;

    end;

  // Wartosci osi rzednych
  f := (FSigmaMax-FSigmaMin) / (FHeight/50);
  for i := Round(FSigmaMin / f) to Round(FSigmaMax / f) do
    begin

    point := Classes.Point(0, Round((1-((i*f-FSigmaMin) / (FSigmaMax-FSigmaMin)))*(FHeight*FSamples-1)));

    if (point.Y >= 0) and (point.Y <= FHeight*FSamples-1) then
      begin
      Buffer.Canvas.MoveTo(axes.X, point.Y);
      Buffer.Canvas.LineTo(axes.X+8*FSamples, point.Y);
      s := FloatToStrF(i*f, ffGeneral, 3, 2);
      Buffer.Canvas.TextOut(axes.X+12*FSamples, point.Y-(Buffer.Canvas.TextHeight(s) div 2), s);
      end;

    end;

  end;

end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.DrawDataSeries(Buffer: TBitmap);
begin

if FShowDataSeries then
  inherited DrawDataSeries(Buffer);

end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.DrawLegend(Buffer: TBitmap; Top: Integer = 0);
const
  BLOCKS_COUNT = 6;
  BLOCK_WIDTH = 22;
  BLOCK_HEIGHT = 16;
var
  b: Byte;
  x, y: Integer;
  s: String;
begin

// Parametry czcionki
Buffer.Canvas.Font.Color := WAVELET_THEMES[FColorTheme].FontColor;
Buffer.Canvas.Font.Size := Ceil(7*FSamples);

// Parametry piora
Buffer.Canvas.Pen.Color := WAVELET_THEMES[FColorTheme].BorderColor;
Buffer.Canvas.Pen.Width := Ceil(0.5*FSamples);

// Skala
for b := 0 to BLOCKS_COUNT-1 do
  begin

  // Dane
  x := Buffer.Width - ((BLOCKS_COUNT-b)*(BLOCK_WIDTH-1)+8)*FSamples;
  y := 8*FSamples;
  s := FloatToStrF(b/(BLOCKS_COUNT-1), ffFixed, 2, 1);

  // Blok
  Buffer.Canvas.Brush.Style := bsSolid;
  Buffer.Canvas.Brush.Color := WAVELET_THEMES[FColorTheme].Colors[Floor((b/(BLOCKS_COUNT-1))*255)];
  Buffer.Canvas.Rectangle(Bounds(x, y, BLOCK_WIDTH*FSamples, BLOCK_HEIGHT*FSamples));

  // Skala
  Buffer.Canvas.Brush.Style := bsClear;
  Buffer.Canvas.TextOut(x+(BLOCK_WIDTH*FSamples-Buffer.Canvas.TextWidth(s)) div 2, y+(BLOCK_HEIGHT*FSamples-Buffer.Canvas.TextHeight(s)) div 2, s);

  end;

// Serie danych
if FShowDataSeries then
  inherited DrawLegend(Buffer, y + BLOCK_HEIGHT*FSamples);

end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.PixelsToPoint(Left, Top: Integer; var X, Y: Double);
begin
PixelsToValue(
  Left, Top,
  FAxisXMin, FAxisXMax,
  FSigmaMin, FSigmaMax,
  X, Y
  );
end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.LoadFromStream(Stream: TStream);
var
  buffer: TMPEBuffer;
begin

inherited LoadFromStream(Stream);

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Dane wykresu
FWavelet := TMPEWavelet(buffer.ReadByte);
FSigmaMin := buffer.ReadDouble;
FSigmaMax := buffer.ReadDouble;
FInterpolationLevel := buffer.ReadDouble;
FColorTheme := buffer.ReadInteger;
FColorsNumber := buffer.ReadWord;
FShowDataSeries := buffer.ReadBoolean;

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
procedure TMPEWaveletGraph.SaveToStream(Stream: TStream);
var
  buffer: TMPEBuffer;
begin

inherited SaveToStream(Stream);

// Tworzenie bufora
buffer := TMPEBuffer.Create(Stream);

// Dane wykresu
buffer.WriteData(Byte(FWavelet));
buffer.WriteData(FSigmaMin);
buffer.WriteData(FSigmaMax);
buffer.WriteData(FInterpolationLevel);
buffer.WriteData(FColorTheme);
buffer.WriteData(FColorsNumber);
buffer.WriteData(FShowDataSeries);

// Zwalnianie bufora
buffer.Free;

end;

// -----------------------------------------------------------------------------
constructor TMPEWaveletGraph.Create;
begin

inherited Create;

// Parametry
FWavelet := wlMexicanHat;
FSigmaMin := 0.1;
FSigmaMax := 10;
FInterpolationLevel := 1;
FColorTheme := 0;
FColorsNumber := 32;
FShowDataSeries := True;

end;

end.
